<?php

namespace Drupal\ai_migration;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai_migration\Service\AiMigrationCacheProviderInterface;
use Drupal\ai_migration\Service\AiMigrationPromptManager;
use Drupal\Core\Http\ClientFactory;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\schemata\SchemaFactory;
use Symfony\Component\Serializer\Serializer;

/**
 * Class to handle AI migrations.
 *
 * This class is responsible for converting URLs
 * to entities using the AI provider.
 */
class AiMigrator {

  /**
   * The schema format for Schemata and Drupal's core serialization.
   */
  private const SCHEMA_FORMAT = 'schema_json:api_json';

  /**
   * The type of AI operation to be performed.
   */
  private const OPERATION_TYPE = 'chat';

  /**
   * The AI provider plugin manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  public $aiProviderPluginManager;

  /**
   * The ai migrator cache provider.
   *
   * @var \Drupal\ai_migration\Service\AiMigrationCacheProviderInterface
   */
  protected $cache;

  /**
   * Drupal HTTP client.
   *
   * @var \Drupal\Core\Http\ClientFactory
   */
  protected $httpClient;

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The ID of the AI provider we are using.
   *
   * @var string
   */
  private $providerId;

  /**
   * The model ID we are using.
   *
   * @var string
   */
  private $modelId;

  /**
   * The URL being processed.
   *
   * @var string
   */
  public $url = '';

  /**
   * The serializer responsible for serializing data before sending to provider.
   *
   * @var \Symfony\Component\Serializer\Serializer
   */
  protected Serializer $serializer;

  /**
   * The schemata factory service.
   *
   * @var \Drupal\schemata\SchemaFactory
   */
  protected SchemaFactory $schemaFactory;

  /**
   * Constructs a new AiMigrator.
   */
  public function __construct(
    AiProviderPluginManager $aiProviderPluginManager,
    AiMigrationCacheProviderInterface $aiMigrationCacheProvider,
    LoggerChannelFactoryInterface $loggerChannelFactory,
    ClientFactory $httpClientFactory,
    Serializer $serializer,
    SchemaFactory $schemaFactory,
  ) {
    // Set the HTTP client with a timeout and error handling.
    $this->httpClient = $httpClientFactory->fromOptions([
      'timeout' => 30,
    ]);

    // Set the AI provider plugin manager.
    $this->aiProviderPluginManager = $aiProviderPluginManager;

    // Set the logger service, too.
    $this->logger = $loggerChannelFactory->get('ai_migration');

    // @todo Set the AI migration cache provider based on config.
    $this->cache = $aiMigrationCacheProvider;

    // Set the serializer service.
    $this->serializer = $serializer;

    // Set the AI migration schema builder.
    $this->schemaFactory = $schemaFactory;
  }

  /**
   * Converts a URL to migratable data using the AI provider.
   *
   * @param \Drupal\ai_migration\Service\AiMigrationPromptManager $prompt_manager
   *   The prompt manager service.
   * @param string $url
   *   The URL for the content we want to migrate.
   * @param string $html
   *   The HTML to migrate if we already have it.
   * @param string $entity_type
   *   The entity type we're migrating into.
   * @param string $bundle
   *   The bundle we're migrating into.
   * @param array $modelConfig
   *   The model configuration for the AI provider.
   *
   * @return array|bool
   *   The converted content or FALSE if conversion fails.
   */
  public function convert(AiMigrationPromptManager $prompt_manager, string $url = '', string $html = '', string $entity_type = '', string $bundle = '', array $modelConfig = []): array|bool {
    $this->url = $url;

    // Get provider and model config.
    $default_provider = $this->aiProviderPluginManager->getDefaultProviderForOperationType(self::OPERATION_TYPE);
    $this->providerId = $modelConfig['provider_id'] ?? $default_provider['provider_id'];
    $this->modelId = $modelConfig['model_id'] ?? $default_provider['model_id'];

    $provider = $this->aiProviderPluginManager->createInstance($this->providerId, [
      'model_id' => $this->modelId,
      'http_client_options' => [
        'timeout' => 300,
      ],
    ]);

    $provider->setConfiguration($modelConfig['config'] ?? []);

    // Create the schema and log it.
    $schema = $this->createSchema($entity_type, $bundle);
    $this->logger->debug('AI migration schema for url @url: <pre>@schema</pre>',
      ['@url' => $url, '@schema' => $schema]);

    // Set system role.
    // @todo Add tokens vs str_replace?
    $system_prompt = str_replace('[ai:migration:schema]', $schema, $prompt_manager->getPrompt('system'));
    $provider->setChatSystemRole($system_prompt);

    $this->logger->debug('AI system prompt for @url: <pre>@prompt</pre>',
      ['@url' => $url, '@prompt' => print_r($system_prompt, TRUE)]);

    // If we don't have html, get it.
    if (empty($html)) {
      try {
        $response = $this->httpClient->request('GET', $url);
        $html = $response->getBody()->getContents();
      }
      catch (\Exception $e) {
        $this->logger->error('Failed to fetch content from URL @url: @message',
          ['@url' => $url, '@message' => $e->getMessage()]);
        return FALSE;
      }
    }

    // @todo Add tokens vs str_replace?
    $user_prompt = str_replace('[ai:migration:content]', $html, $prompt_manager->getPrompt('user'));

    $messages = new ChatInput([
      new ChatMessage('user', $user_prompt),
    ]);

    $cachedResponse = $this->cache->getPromptResponse($user_prompt, $this->providerId, $this->modelId);
    if ($cachedResponse) {
      return $cachedResponse;
    }

    try {
      $response = $provider->chat($messages, $this->modelId,
        ['ai-migration'])->getNormalized()->getText();
    }
    catch (\Exception $e) {
      $this->logger->error('AI migration failed for url @url: @message',
        ['@url' => $url, '@message' => $e->getMessage()]);
      return FALSE;
    }

    // Clean up the response.
    $clean_response = $this->normalizeResponse($response, $bundle);

    // Cache the response for future use.
    $this->cache->setPromptResponse($clean_response, $user_prompt, $this->providerId, $this->modelId);

    // Log for debugging purpose for now.
    // We may want to put this behind a setting later.
    $this->logger->debug('AI migration response for url @url: <pre>@response</pre>',
      ['@url' => $url, '@response' => print_r($clean_response, TRUE)]);

    return $clean_response;
  }

  /**
   * Creates a schema for the content type we're migrating to.
   *
   * @param string $entity_type
   *   The entity type to create the schema for.
   * @param string $bundle
   *   The name of the bundle to create the schema for.
   *
   * @return string
   *   The created schema in JSON format.
   */
  public function createSchema(string $entity_type, string $bundle): string {
    // Generate a Schema object.
    $schema = $this->schemaFactory->create($entity_type, $bundle);

    // Render the schema as a string conforming to the selecting schema type.
    $json_schema = $this->serializer->serialize($schema, self::SCHEMA_FORMAT);

    return $json_schema;
  }

  /**
   * Cleans the response to match the expected format.
   *
   * @param string $response
   *   The response to clean.
   * @param string $bundle
   *   The bundle we're migrating into.
   *
   * @return array
   *   The cleaned response as an array.
   */
  public function normalizeResponse(string $response, string $bundle): array {
    // Clean up the response to match the expected format.
    $cleaned = preg_replace('/```json|`/', '', $response);
    $decoded = json_decode($cleaned, TRUE);

    if ($decoded === NULL) {
      $this->logger->error('Failed to decode cleaned AI JSON response: @response', ['@response' => $response]);
      return [];
    }
    else {
      $this->logger->debug('Decoded AI JSON response: <pre>@decoded</pre>', ['@decoded' => print_r($decoded, TRUE)]);
    }

    // Denormalize using your custom denormalizer.
    try {
      $entity = $this->serializer->denormalize(
        $decoded['data'],
        'Drupal\node\Entity\Node',
        'ai_migration',
        ['resource_type' => 'node--' . $bundle]
      );

      // Return the entity as an array ready for migration.
      return $entity->toArray();
    }
    catch (\Exception $e) {
      $this->logger->error('Denormalization failed for bundle @bundle: @message', [
        '@bundle' => $bundle,
        '@message' => $e->getMessage(),
      ]);
      return [];
    }
  }

}
