<?php

declare(strict_types=1);

namespace Drupal\ai_migration\Enum;

/**
 * Default prompts for AI migration.
 */
enum PromptDefault: string {

  case PromptUser = 'user';
  case PromptSystem = 'system';

  /**
   * Get the prompt text.
   *
   * @return string
   *   The default prompt text.
   */
  public function getPromptText(): string {
    return match ($this) {
      self::PromptUser => $this->getUserPrompt(),
      self::PromptSystem => $this->getSystemPrompt(),
    };
  }

  /**
   * Get the default system prompt.
   *
   * @return string
   *   The default system prompt.
   */
  protected function getSystemPrompt(): string {
    $default = <<<EOT
      As a highly efficient Web Content Extractor, your sole purpose is to analyze the provided HTML and context, extracting the primary article content
      into a structured JSON object according to the rules and schema defined.

      JSON Output Schema Rules:
      1. You must adhere strictly to the following JSON structure.
      2. If a value for a field cannot be found, return null.
      3. For relationship fields, YOU MUST always respond with these sub-fields: type, name
      4. Always include "attributes" objects. If a field is contained within an "attributes" object in the schema, respond with the field embedded in an "attributes" object. For example, if the schema specifies this:

      "attributes":{
          "name":{
              "type":"string",
               "title":"Name",
                "maxLength":255
           }
      }

      you should respond with:

      "attributes": {
          "name": ""
      }

      JSON Schema to be used:
      [ai:migration:schema]
EOT;

    return trim($default);
  }

  /**
   * Get the default user prompt.
   *
   * @return string
   *   The default user prompt.
   */
  protected function getUserPrompt(): string {
    $timestamp = time();

    $default = <<<EOT
      Rules & Context for processing HTML:

      HTML Rules:
      1. Use the HTML structure that was processed but do not place HTML <head> or <body> tags in any element of the schema.

      URL Conversion Rules:
      1. For any URLs that are returned for a field in the schema, you must make sure they are absolute and not relative.

      Example 1 (Root-Relative): If the base URL is https://example.com and an image src is /image1.jpg, the output URL must be https://example.com/image1.jpg.
      Example 2 (Document-Relative): If the page URL is https://example.com/level/ and an image src is sub/image1.jpg, the output URL must be https://example.com/level/sub/image1.jpg.

      Date and Time Rules:
      1. All unix timestamps in the final JSON must be in seconds, not milliseconds.
      2. For the created_at and updated_at fields, use the current date and time in unix timestamp format.

      This is the HTML to be processed:
      [ai:migration:content]
EOT;

    return trim($default);
  }

}
