<?php

declare(strict_types=1);

namespace Drupal\ai_migration;

use Symfony\Component\HtmlSanitizer\HtmlSanitizerConfig;

/**
 * Builder class for HtmlSanitizerConfig.
 */
class HtmlSanitizerConfigBuilder implements HtmlSanitizerConfigBuilderInterface {

  /**
   * The HTML sanitizer configuration.
   */
  protected HtmlSanitizerConfig $config;

  /**
   * Creates a new HtmlSanitizerConfigBuilder instance.
   */
  public function __construct() {
    $this->config = new HtmlSanitizerConfig();
  }

  /**
   * {@inheritdoc}
   */
  public function build(array $options): HtmlSanitizerConfig {
    if (isset($options['allowSafeElements']) && isset($options['allowStaticElements'])) {
      throw new \InvalidArgumentException('Can only choose allowSafeElements OR allowStaticElements as a migrate config.');
    }

    foreach ($options as $key => $value) {
      if (!method_exists($this->config, $key)) {
        throw new \InvalidArgumentException(sprintf('Option "%s" is not supported by html_processor.', $key));
      }

      try {
        $this->processOption($key, $value);
      }
      catch (\Exception $e) {
        throw new \InvalidArgumentException(sprintf('Error processing option "%s": %s', $key, $e->getMessage()));
      }
    }

    return $this->config;
  }

  /**
   * Process a single config option.
   *
   * @param string $key
   *   The option name.
   * @param mixed $value
   *   The option value.
   */
  protected function processOption(string $key, $value): void {
    switch ($key) {
      case 'allowSafeElements':
      case 'allowStaticElements':
      case 'forceHttpsUrls':
      case 'allowRelativeLinks':
      case 'allowRelativeMedias':
        if ($value === TRUE) {
          $this->config = $this->config->{$key}($value);
        }
        break;

      case 'allowElement':
      case 'blockElement':
      case 'dropElement':
      case 'allowAttribute':
      case 'dropAttribute':
      case 'allowedLinkSchemes':
      case 'allowedLinkHosts':
      case 'allowedMediaSchemes':
      case 'allowedMediaHosts':
        $this->processArrayOptions($key, $value);
        break;

      case 'forceAttribute':
        if (is_array($value) && count($value) === 3) {
          $this->processForceAttribute($value);
        }
        else {
          throw new \InvalidArgumentException('Incorrect parameters supplied for forceAttribute.');
        }
        break;

      default:
        throw new \InvalidArgumentException(sprintf('Unsupported option: %s', $key));
    }
  }

  /**
   * Processes an array of options for the sanitizer.
   *
   * @param string $method
   *   The method to call on the sanitizer config.
   * @param array|string $values
   *   The values to process.
   */
  protected function processArrayOptions(string $method, array|string $values): void {
    if (is_array($values) && count(array_keys($values)) === 2) {
      if (!is_string($values[0])) {
        throw new \InvalidArgumentException('The first value of the config must be a string.');
      }

      $option_key = $values[0];
      $option_values = $values[1];
      $this->config = $this->config->{$method}($option_key, $option_values);
    }
    else {
      foreach ($values as $value) {
        $this->config = $this->config->{$method}($value);
      }
    }
  }

  /**
   * Process forceAttribute option.
   *
   * @param array $value
   *   The value containing element, attribute, and value.
   */
  protected function processForceAttribute(array $value): void {
    if (count($value) !== 3) {
      throw new \InvalidArgumentException('forceAttribute requires exactly 3 parameters: element, attribute, and value');
    }

    $this->config = $this->config->forceAttribute($value[0], $value[1], $value[2]);
  }

}
