<?php

namespace Drupal\ai_migration\Normalizer;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\jsonapi\ResourceType\ResourceType;
use Drupal\serialization\Normalizer\NormalizerBase;
use Symfony\Component\Serializer\Exception\InvalidArgumentException;
use Symfony\Component\Serializer\Normalizer\DenormalizerInterface;

/**
 * Denormalizer for AI migration JSON:API-like data to Drupal entities.
 */
class AiMigrationDenormalizer extends NormalizerBase implements DenormalizerInterface {

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructor.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityRepositoryInterface $entityRepository,
    LoggerChannelFactoryInterface $loggerChannelFactory,
  ) {
    $this->logger = $loggerChannelFactory->get('ai_migration');
  }

  /**
   * {@inheritdoc}
   */
  public function supportsNormalization($data, ?string $format = NULL, array $context = []): bool {
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function normalize($object, $format = NULL, array $context = []): array|string|int|float|bool|\ArrayObject|NULL {
    throw new \LogicException('This method should never be called.');
  }

  /**
   * {@inheritdoc}
   */
  protected function prepareInput(array $data, ResourceType $resource_type, $format, array $context): array {
    // Modify the input data as needed for entity creation.
    return $data;
  }

  /**
   * {@inheritdoc}
   */
  public function denormalize($data, string $type, ?string $format = NULL, array $context = []): EntityInterface {
    if (!$this->supportsDenormalization($data, $type, $format, $context)) {
      throw new InvalidArgumentException('The data cannot be denormalized.');
    }

    // Extract entity type and bundle from the type string.
    [$entity_type, $bundle] = explode('--', $context['resource_type'] ?? $type);

    if (!$entity_type || !$bundle) {
      throw new InvalidArgumentException('Entity type and bundle must be specified in context.');
    }

    // Process the JSON:API data structure.
    $processed_data = $this->processJsonApiData($data, $bundle);

    // Create entity storage.
    $storage = $this->entityTypeManager->getStorage($entity_type);

    // Create and return the entity.
    return $storage->create($processed_data);
  }

  /**
   * {@inheritdoc}
   */
  public function supportsDenormalization($data, string $type, ?string $format = NULL, array $context = []): bool {
    return $format === 'ai_migration' && is_array($data);
  }

  /**
   * Processes JSON:API data structure into Drupal entity format.
   *
   * @param array $data
   *   The JSON:API data structure.
   * @param string $bundle
   *   The bundle.
   *
   * @return array
   *   Processed data ready for entity creation.
   */
  protected function processJsonApiData(array $data, string $bundle): array {
    $processed = ['type' => $bundle];

    // Check if this is a full JSON:API document with 'data' wrapper.
    if (isset($data['data'])) {
      $entity_data = $data['data'];
    }
    else {
      $entity_data = $data;
    }

    // Process attributes.
    if (isset($entity_data['attributes'])) {
      foreach ($entity_data['attributes'] as $field_name => $field_value) {
        // Skip Drupal internal fields that shouldn't be set during creation.
        if (in_array($field_name, [
          'drupal_internal__nid',
          'drupal_internal__vid',
          'revision_timestamp',
          'created',
          'changed',
        ])) {
          continue;
        }

        $processed[$field_name] = $this->processFieldValue($field_name, $field_value);
      }
    }

    // Process relationships.
    if (isset($entity_data['relationships'])) {
      foreach ($entity_data['relationships'] as $field_name => $relationship) {
        // Skip system relationships that are auto-managed.
        if (in_array($field_name, ['node_type', 'revision_uid'])) {
          continue;
        }
        if (is_array($relationship)) {
          $processed[$field_name] = $this->processRelationshipField($relationship);
        }
      }
    }

    // Debug logging to see what we're processing.
    $this->logger->debug('Processed entity data: @data', [
      '@data' => json_encode($processed, JSON_PRETTY_PRINT),
    ]);

    return $processed;
  }

  /**
   * Processes the value of a specific field during denormalization.
   *
   * @param string $field_name
   *   The name of the field being processed.
   * @param mixed $field_value
   *   The value of the field to process.
   *
   * @return mixed
   *   The processed field value.
   */
  protected function processFieldValue(string $field_name, $field_value) {
    // Handle text fields with format.
    if (is_array($field_value) && isset($field_value['value'])) {
      return $field_value;
    }

    // Handle language code objects.
    if ($field_name === 'langcode' && is_array($field_value) && isset($field_value['value'])) {
      return $field_value['value'];
    }

    // Handle path objects.
    if ($field_name === 'path' && is_array($field_value)) {
      return $field_value;
    }

    // Handle simple values.
    return $field_value;
  }

  /**
   * Processes a relationship field and returns the processed data.
   *
   * This method takes an array representing a relationship field and performs
   * necessary processing to transform or normalize the data as required by the
   * migration logic.
   *
   * @param array $relationship
   *   The relationship field data to be processed.
   *
   * @return array
   *   The processed relationship field data.
   */
  protected function processRelationshipField(array $relationship): array {
    if (!isset($relationship['data'])) {
      return [];
    }

    // Handle single relationship.
    if (isset($relationship['data']['type'])) {
      return $this->processEntityReference($relationship['data']);
    }

    // Handle multiple relationships.
    if (is_array($relationship['data'])) {
      $processed = [];
      foreach ($relationship['data'] as $reference) {
        if (isset($reference['type'])) {
          $entity_ref = $this->processEntityReference($reference);
          if (!empty($entity_ref)) {
            $processed[] = $entity_ref;
          }
        }
      }
      return $processed;
    }

    return [];
  }

  /**
   * Processes an entity reference array and returns the processed result.
   *
   * This method handles the transformation or normalization of entity reference
   * data structures as required by the migration process.
   *
   * @param array $reference
   *   The entity reference array to process.
   *
   * @return array
   *   The processed entity reference array.
   */
  protected function processEntityReference(array $reference): array {
    // @todo Extend this method to handle different entity types as needed.
    // Extract entity type from JSON:API type.
    [$entity_type, $bundle] = explode('--', $reference['type']);

    // Get the storage handler for taxonomy terms.
    $storage = $this->entityTypeManager->getStorage($entity_type);

    switch ($entity_type) {
      case 'taxonomy_term':
        // Load the term(s) by name and vocabulary.
        $entities = $storage->loadByProperties([
          'name' => $reference['attributes']['name'],
          'vid' => $bundle,
        ]);
        break;

      default:
        $this->logger->warning('Unsupported entity type in reference: @type @name', [
          '@type' => $reference['type'],
          '@name' => $reference['attributes']['name'],
        ]);
        return [];
    }

    if (!empty($entities)) {
      // Load the first entity from the results.
      $entity = reset($entities);
      return ['target_id' => $entity->id()];
    }
    else {
      // Handle the case where no term was found.
      $this->logger->warning('Referenced entity not found: @type', [
        '@type' => $reference['type'],
      ]);
      return [];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedTypes(?string $format): array {
    return [
      '*' => $format === 'ai_migration',
    ];
  }

}
