<?php

namespace Drupal\ai_migration\Normalizer\jsonapi;

use Drupal\Component\Utility\NestedArray;
use Drupal\jsonapi\ResourceType\ResourceType;
use Drupal\jsonapi\ResourceType\ResourceTypeRepository;
use Drupal\schemata\SchemaUrl;
use Drupal\schemata_json_schema\Normalizer\jsonapi\SchemataSchemaNormalizer;
use Symfony\Component\Serializer\Normalizer\NormalizerInterface;

/**
 * AI Migration override of SchemataSchemaNormalizer.
 *
 * This is largely a copy of the parent class with modifications to
 * filter out fields we don't need in the schema -- that is, AI
 * won't care about it, and it will just confuse the AI.
 */
class AiMigrationSchemaNormalizer extends SchemataSchemaNormalizer {

  /**
   * Constructs a new AiMigrationSchemaNormalizer object.
   *
   * @param \Drupal\jsonapi\ResourceType\ResourceTypeRepository $resourceTypeRepository
   *   The resource type repository service.
   */
  public function __construct(protected ResourceTypeRepository $resourceTypeRepository) {
  }

  /**
   * {@inheritdoc}
   */
  public function normalize($entity, $format = NULL, array $context = []): array|bool|string|int|float|null|\ArrayObject {
    /** @var \Drupal\schemata\Schema\SchemaInterface $entity */
    $generated_url = SchemaUrl::fromSchema($this->format, $this->describedFormat, $entity)
      ->toString(TRUE);
    // Create the array of normalized fields, starting with the URI.
    /** @var \Drupal\jsonapi\ResourceType\ResourceTypeRepository $resource_type_repository */
    $resource_type = $this->resourceTypeRepository->get(
      $entity->getEntityTypeId(),
      $entity->getBundleId() ?: $entity->getEntityTypeId()
    );
    $normalized = [
      '$schema' => 'http://json-schema.org/draft-06/schema#',
      'title' => 'JSON:API Schema',
      'description' => 'This is a schema for responses in the JSON:API format. For more, see http://jsonapi.org',
      'id' => $generated_url->getGeneratedUrl(),
      'type' => 'object',
      'required' => [
        'data',
      ],
      'properties' => [
        'data' => [
          'description' => '\"Resource objects\" appear in a JSON:API document to represent resources.',
          'type' => 'object',
          'required' => [
            'type',
          ],
          'properties' => [
            'type' => [
              'type' => 'string',
              'title' => 'type',
              'description' => $this->t('Resource type'),
              'enum' => [$resource_type->getTypeName()],
            ],
            'attributes' => [
              'description' => 'Members of the attributes object (\'attributes\") represent information about the resource object in which it\'s defined . ',
              'type' => 'object',
              'additionalProperties' => FALSE,
            ],
            'relationships' => [
              'description' => 'Members of the relationships object(\'relationships\") represent references from the resource object in which it\'s defined to other resource objects . ',
              'type' => 'object',
              'additionalProperties' => FALSE,
            ],
          ],
          'additionalProperties' => FALSE,
        ],
        'jsonapi' => [
          'description' => 'An object describing the server\'s implementation',
          'type' => 'object',
          'properties' => [
            'version' => [
              'type' => 'string',
            ],
            'meta' => [
              'description' => 'Non-standard meta-information that can not be represented as an attribute or relationship.',
              'type' => 'object',
              'additionalProperties' => TRUE,
            ],
          ],
          'additionalProperties' => FALSE,
        ],
      ],
      'additionalProperties' => TRUE,
    ];

    // Stash schema request parameters.
    $context['entityTypeId'] = $entity->getEntityTypeId();
    $context['bundleId'] = $entity->getBundleId();
    $context['resourceType'] = $resource_type;

    // Retrieve 'properties' and possibly 'required' nested arrays.
    $schema_overrides = [
      'properties' => [
        'data' => $this->normalizeJsonapiProperties(
          $this->getProperties($entity, $format, $context),
          $format,
          $context
        ),
      ],
    ];
    return NestedArray::mergeDeep($normalized, $entity->getMetadata(), $schema_overrides);
  }

  /**
   * {@inheritdoc}
   */
  protected function normalizeJsonapiProperties(array $items, $format, array $context = []) {
    $normalized = [];
    $resource_type = $context['resourceType'];
    assert($resource_type instanceof ResourceType);
    assert($this->serializer instanceof NormalizerInterface);

    // Filter out fields that are not relevant for AI migration.
    foreach ($items as $name => $property) {
      if (!$resource_type->isFieldEnabled($name)) {
        continue;
      }

      $filter = [
        'id',
        'vid',
        'uid',
        'langcode',
        'default_langcode',
        'status',
        'sticky',
        'promote',
        'revision_default',
        'revision_translation_affected',
        'revision_timestamp',
        'revision_created',
        'revision_log',
        'revision_log_message',
        'revision_uid',
        'weight',
        'path',
        'links',
        'node_type',
        'drupal_internal__nid',
        'drupal_internal__vid',
        'drupal_internal__tid',
        'drupal_internal__revision_id',
      ];

      if (in_array($name, $filter)) {
        continue;
      }

      $context['name'] = $resource_type->getPublicName($name);
      $item = $this->serializer->normalize($property, $format, $context);
      if (!empty($item)) {
        $normalized = NestedArray::mergeDeep($normalized, $item);
      }
    }

    return $normalized;
  }

  /**
   * {@inheritdoc}
   */
  public static function getPriority(): int {
    // Higher priority than parent to ensure this gets used.
    return 50;
  }

}
