<?php

namespace Drupal\ai_migration\Normalizer\jsonapi;

use Drupal\Core\Field\FieldTypePluginManagerInterface;
use Drupal\Core\Field\Plugin\Field\FieldType\EntityReferenceItem;
use Drupal\schemata\SchemaFactory;
use Drupal\schemata_json_schema\Normalizer\jsonapi\RelationshipFieldDefinitionNormalizer as SchemataNormalizer;
use Symfony\Component\Serializer\SerializerInterface;

/**
 * Normalizer for RelationshipFieldDefinitionNormalizer entities.
 *
 * This normalizes the JSON API relationships. This normalizer shortcuts the
 * recursion for the entity reference field. A JSON API relationship is what it
 * is, regardless of how Drupal stores the relationship.
 */
class RelationshipFieldDefinitionNormalizer extends SchemataNormalizer {

  /**
   * Constructs a RelationshipFieldDefinitionNormalizer object.
   *
   * @param \Drupal\Core\Field\FieldTypePluginManagerInterface $field_type_manager
   *   The field type plugin manager.
   * @param \Drupal\schemata\SchemaFactory $schemaFactory
   *   The schema factory service.
   */
  public function __construct(
    protected FieldTypePluginManagerInterface $field_type_manager,
    protected SchemaFactory $schemaFactory,
  ) {
    parent::__construct($field_type_manager);
  }

  /**
   * {@inheritdoc}
   *
   * This method is called by the Serializer service *after* construction.
   * We also call the parent's setSerializer to ensure it also has it.
   */
  public function setSerializer(SerializerInterface $serializer): void {
    $this->serializer = $serializer;
    parent::setSerializer($serializer);
  }

  /**
   * {@inheritdoc}
   */
  public function supportsNormalization($data, $format = NULL, array $context = []): bool {
    // We only want to work with fields created by users.
    if (!str_starts_with($context['name'], 'field_')) {
      return FALSE;
    }

    $type = $data->getItemDefinition()->getFieldDefinition()->getType();
    $class = $this->fieldTypeManager->getPluginClass($type);
    return $class == EntityReferenceItem::class || is_subclass_of($class, EntityReferenceItem::class);
  }

  /**
   * {@inheritdoc}
   */
  public function normalize($entity, $format = NULL, array $context = []): array|bool|string|int|float|null|\ArrayObject {
    /** @var \Drupal\Core\Field\FieldDefinitionInterface $entity */
    $target_entity_type = $entity->getFieldStorageDefinition()->getSetting('target_type');
    $handler_settings = $entity->getSetting('handler_settings');
    $target_bundles = $handler_settings['target_bundles'] ?? NULL;

    // If all bundles are allowed or it's a bundle-less entity, we can't
    // reliably inline one schema. You could choose to return the base
    // entity schema or, as we do here, fall back to the default behavior.
    if (empty($target_bundles) || count($target_bundles) > 1) {
      // Fallback: Just return the default linkage entity schema.
      return parent::normalize($entity, $format, $context);
    }

    $target_bundle = reset($target_bundles);

    // --- Circular Reference Protection ---
    // We use the $context array to track our recursion path.
    $processed_key = "{$target_entity_type}:{$target_bundle}";
    if (isset($context[$processed_key])) {
      return [
        'title' => "Circular Reference ({$processed_key})",
        'description' => 'Schema inlining stopped to prevent recursion.',
        'type' => 'entity',
      ];
    }
    // Add this entity to the context for the *next* normalization call.
    $context[$processed_key] = TRUE;

    try {
      // 1. Create the schema entity for the SUB-ENTITY.
      $sub_schema = $this->schemaFactory->create($target_entity_type, $target_bundle);

      // 2. Normalize the SUB-SCHEMA.
      // This is recursive: if this sub-schema has relationships,
      // the serializer will find this normalizer again.
      // We pass the $context to handle circular references.
      $normalized_sub_schema = [
        'properties' => [
          'relationships' => [
            'description' => $this->t('Entity relationships'),
            'properties' => [$context['name'] => $this->serializer->normalize($sub_schema, $format, $context)],
            'type' => 'object',
          ],
        ],
      ];

      return $normalized_sub_schema;

    }
    catch (\Exception $e) {
      // Something went wrong, return a simple entity.
      return [
        'type' => 'entity',
        'title' => 'Error',
        'description' => 'Could not inline sub-schema: ' . $e->getMessage(),
      ];
    }
  }

}
