<?php

declare(strict_types=1);

namespace Drupal\ai_migration\PluginManager;

use Drupal\ai_migration\Attribute\AiPrompt;
use Drupal\ai_migration\Plugin\ai_migration\prompt\AiPromptInterface;
use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;

/**
 * Manages AI Prompt plugins and contains the core processing logic.
 */
class PromptPluginManager extends DefaultPluginManager {

  /**
   * The prompt plugins.
   *
   * @var \Drupal\ai_migration\Plugin\ai_migration\prompt\AiPromptInterface[]
   */
  protected $plugins;

  /**
   * Constructs a new AiPromptManager object.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler) {
    parent::__construct(
      'Plugin/ai_migration/prompt',
      $namespaces,
      $module_handler,
      AiPromptInterface::class,
      AiPrompt::class
    );
    $this->alterInfo('ai_migration_prompt_info');
    $this->setCacheBackend($cache_backend, 'ai_migration_prompt_plugins');
  }

  /**
   * Searches the plugins to find a match by type value.
   *
   * @param string $type
   *   The type value to search for.
   *
   * @return string|null
   *   The matching plugin ID, or NULL if no match is found.
   */
  public function findPlugIdByType(string $type): ?string {
    foreach ($this->getDefinitions() as $id => $definition) {
      if (!empty($definition['type']) && $definition['type'] === $type) {
        return $id;
      }
    }

    return NULL;
  }

  /**
   * Searches the plugins to find the default plugin ID.
   *
   * @return string|null
   *   The default plugin ID, or NULL if no default plugin is found.
   */
  public function findDefaultPluginId(): ?string {
    foreach ($this->getDefinitions() as $id => $definition) {
      if (!empty($definition['is_default']) && $definition['is_default'] === TRUE) {
        return $id;
      }
    }
    return NULL;
  }

  /**
   * Get the plugin ID to search for based on the type.
   *
   * @param string|null $type
   *   The type of prompt to get.
   *
   * @return string
   *   The plugin ID.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function getPluginIdByType(?string $type): string {
    if ($type) {
      // Search all plugin definitions for one that matches the type.
      $plugin_id = $this->findPlugIdByType($type);
      if (!$plugin_id) {
        throw new PluginException(sprintf('The AI Prompt plugin for type "%s" was not found.', $type));
      }
    }
    else {
      // Search for a default plugin only if no type was specified.
      $plugin_id = $this->findDefaultPluginId();
    }

    if (!$plugin_id) {
      throw new PluginException('No suitable AI Prompt plugin found for the provided configuration.');
    }

    return $plugin_id;
  }

  /**
   * Get the plugin instance based on the type.
   *
   * @param string|null $type
   *   The type of prompt to get.
   *
   * @return object
   *   The plugin instance.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function getPluginByType(?string $type): object {
    $plugin_id = $this->getPluginIdByType($type);
    return $this->createInstance($plugin_id);
  }

  /**
   * Get the default plugin instance.
   *
   * @return object
   *   The default plugin instance.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function getDefaultPlugin(): object {
    $plugin_id = $this->findDefaultPluginId();
    return $this->createInstance($plugin_id);
  }

}
