<?php

declare(strict_types=1);

namespace Drupal\ai_migration\Service;

use Drupal\Core\Cache\CacheBackendInterface;

/**
 * Provides a cache service for AI migration responses.
 */
abstract class AiMigrationCacheProviderBase implements AiMigrationCacheProviderInterface {

  public const CACHE_BASE_CID_PREFIX = 'ai_migrator';

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */

  /**
   * Create a unique hash based on a string, provider, and model.
   *
   * @param string $string
   *   A string, such as prompt,to be hashed for caching.
   * @param string $provider_id
   *   The ID of the AI provider.
   * @param string $model_id
   *   The ID of the AI model being used.
   *
   * @return string
   *   The generated hash as a 32-bit xxHash value.
   */
  protected function hashContent(string $string, string $provider_id, string $model_id): string {
    return hash('xxh32', $string . $provider_id . $model_id);
  }

  /**
   * Create a unique cache ID based on a prefix using the based id.
   *
   * @param string $prefix
   *   The prefix to use for the cache ID.
   * @param string $unique_id
   *   A unique identifier.
   *
   * @return string
   *   The generated cache ID.
   */
  protected function buildCacheId(string $prefix, string $unique_id): string {
    return sprintf('%s:%s:%s', self::CACHE_BASE_CID_PREFIX, $prefix, $unique_id);
  }

  /**
   * {@inheritdoc}
   */
  abstract public function getPromptCacheId(string $prompt, string $provider_id, string $model_id): string;

  /**
   * {@inheritdoc}
   */
  abstract public function getPromptResponse(string $prompt, string $provider_id, string $model_id): ?array;

  /**
   * {@inheritdoc}
   */
  abstract public function setPromptResponse(array $response, string $prompt, string $provider_id, string $model_id, int $expire = CacheBackendInterface::CACHE_PERMANENT): void;

}
