<?php

namespace Drupal\ai_migration\Service;

use Drupal\Core\Cache\CacheBackendInterface;

/**
 * Interface for a AI migration response cache service.
 */
interface AiMigrationCacheProviderInterface {

  /**
   * Generates a unique cache key for storing AI responses.
   *
   * @param string $prompt
   *   The input prompt text to be sent to the AI provider.
   * @param string $provider_id
   *   The ID of the AI provider.
   * @param string $model_id
   *   The ID of the AI model being used.
   *
   * @return string
   *   The generated cache key as a 32-bit xxHash value.
   */
  public function getPromptCacheId(string $prompt, string $provider_id, string $model_id): string;

  /**
   * Retrieves a cached response for the given prompt.
   *
   * @param string $prompt
   *   The prompt to retrieve the cached response for.
   * @param string $provider_id
   *   The ID of the AI provider.
   * @param string $model_id
   *   The ID of the AI model being used.
   *
   * @return array|null
   *   The cached response or NULL if not found.
   */
  public function getPromptResponse(string $prompt, string $provider_id, string $model_id): ?array;

  /**
   * Set the cache for the response.
   *
   * @param array $response
   *   The response to cache.
   * @param string $prompt
   *   The prompt that was used to generate the response.
   * @param string $provider_id
   *   The ID of the AI provider.
   * @param string $model_id
   *   The ID of the AI model being used.
   * @param int $expire
   *   (optional) Expiration time.
   */
  public function setPromptResponse(array $response, string $prompt, string $provider_id, string $model_id, int $expire = CacheBackendInterface::CACHE_PERMANENT): void;

}
