<?php

declare(strict_types=1);

namespace Drupal\ai_migration\Service;

use Drupal\ai_migration\Enum\PromptDefault;
use Drupal\ai_migration\Enum\PromptOperation;
use Drupal\ai_migration\Enum\PromptRole;
use Drupal\ai_migration\Plugin\ai_migration\prompt\AiPromptInterface;
use Drupal\ai_migration\PluginManager\PromptPluginManager;

/**
 * Manages AI Prompts and contains the core processing logic.
 */
class AiMigrationPromptManager {

  const MIGRATION_ROOT_KEY = 'prompt';

  const DEFAULT_OPERATION = 'replace';

  /**
   * The configuration array.
   *
   * @var array
   */
  protected $config = [];

  /**
   * The plugin manager.
   *
   * @var \Drupal\ai_migration\PluginManager\PromptPluginManager
   */
  protected $pluginManager;

  /**
   * Constructs a new AiPromptManager object.
   *
   * @param \Drupal\ai_migration\PluginManager\PromptPluginManager $plugin_manager
   *   The plugin manager.
   */
  public function __construct(PromptPluginManager $plugin_manager) {
    $this->pluginManager = $plugin_manager;
  }

  /**
   * Sets the configuration array.
   *
   * @param array $config
   *   The configuration array.
   */
  public function setConfig(array $config): void {
    $this->config = $config;
  }

  /**
   * Get the prompt through matching plugin based on configuration.
   *
   * @param string $role
   *   The role the prompt is being generated.
   * @param array $migration_config
   *   The configuration array from migration with the prompt configuration.
   *
   * @return string
   *   The final, processed prompt text.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   * @throws \InvalidArgumentException
   */
  public function getPrompt(string $role, array $migration_config = []): string {
    $config = $this->config ?: $migration_config;
    $prompt_config = NULL;

    $this->validateRole($role);

    // Get the configuration for the specified role from the migration config.
    foreach ($config as $role_config) {
      if ($role_config['role'] === $role) {
        $prompt_config = $role_config;
        break;
      }
    }

    // If the prompt configuration is not found for the specific role,
    // return the default prompt for the specific role.
    if (empty($prompt_config)) {
      return PromptDefault::from($role)->getPromptText();
    }

    // Default to replace if not set.
    $operation = $prompt_config['operation'] ?? self::DEFAULT_OPERATION;
    $this->validateOperation($operation);

    $type = $prompt_config['type'] ?? NULL;

    /** @var \Drupal\ai_migration\Plugin\ai_migration\prompt\AiPromptInterface $plugin_instance */
    $plugin_instance = $this->getPluginByType($type);

    $prompt = $plugin_instance->getPrompt($prompt_config);
    $eol = (bool) ($prompt_config['operation_eol'] ?? TRUE);
    return PromptOperation::from($operation)->modify(PromptDefault::from($role)->getPromptText(), $prompt, $eol);
  }

  /**
   * Validates the role and throw an exception if invalid.
   *
   * @param string $role
   *   The role to validate.
   *
   * @throws \InvalidArgumentException
   */
  protected function validateRole(string $role): void {
    // Validate the role. An empty role is not allowed.
    if (!PromptRole::isValid($role)) {
      throw new \InvalidArgumentException(sprintf(
        'Invalid role "%s". Valid roles are: %s',
        $role,
        implode(', ', PromptRole::getValidItems())
      ));
    }
  }

  /**
   * Get a plugin by type.
   *
   * @param string|null $type
   *   The type of plugin to get.
   *
   * @return \Drupal\ai_migration\Plugin\ai_migration\prompt\AiPromptInterface
   *   The plugin instance, returns the default plugin if no type.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function getPluginByType(?string $type): AiPromptInterface {
    return $this->pluginManager->getPluginByType($type);
  }

  /**
   * Get the default plugin.
   *
   * @return \Drupal\ai_migration\Plugin\ai_migration\prompt\AiPromptInterface
   *   The default plugin.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function getDefaultPlugin(): AiPromptInterface {
    return $this->pluginManager->getDefaultPlugin();
  }

  /**
   * Validates the operation and throw an exception if invalid.
   *
   * @param string $operation
   *   The operation to validate.
   *
   * @throws \InvalidArgumentException
   */
  protected function validateOperation(string $operation): void {
    // Validate the operation.
    if (!PromptOperation::isValid($operation)) {
      throw new \InvalidArgumentException(sprintf(
        'Invalid operation "%s". Valid operations are: %s',
        $operation,
        implode(', ', PromptOperation::getValidItems())
      ));
    }
  }

}
