<?php

namespace Drupal\Tests\ai_migration\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\NodeType;
use Drupal\ai_migration\AiMigrator;

/**
 * Tests the AiMigrator class.
 *
 * @group ai_migration
 */
class AiMigratorTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'node',
    'file',
    'user',
    'serialization',
    'jsonapi',
    'ai',
    'ai_migration',
    'schemata',
    'schemata_json_schema',
  ];

  /**
   * The AI provider plugin manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $aiProviderPluginManager;

  /**
   * The AiMigrator instance being tested.
   *
   * @var \Drupal\ai_migration\AiMigrator
   */
  protected $aiMigrator;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install required configuration.
    $this->installConfig(['ai', 'ai_migration']);

    // Add article content type.
    $this->installEntitySchema('node');
    $node_type = NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ]);
    $node_type->save();

    // Get the AI provider plugin manager service.
    $this->aiProviderPluginManager = $this->container->get('ai.provider');

    // Get the logger service.
    $loggerChannelFactory = $this->container->get('logger.factory');

    // Get the HTTP client factory service.
    $httpClientFactory = $this->container->get('http_client_factory');

    // Get the cache service.
    $cache = $this->container->get('ai_migration.cache_provider');

    // Get the serializer service.
    $serializer = $this->container->get('serializer');

    // Get the schemata schema factory service.
    $schemaFactory = $this->container->get('schemata.schema_factory');

    // Create the AiMigrator instance.
    $this->aiMigrator = new AiMigrator($this->aiProviderPluginManager, $cache, $loggerChannelFactory, $httpClientFactory, $serializer, $schemaFactory);
  }

  /**
   * Tests the createSchema method.
   */
  public function testCreateSchema(): void {
    // Build a schema for our bundle.
    $entity_type = 'node';
    $bundle = 'article';

    $schema = $this->aiMigrator->createSchema($entity_type, $bundle);
    $json_schema = json_decode($schema, TRUE);

    $this->assertIsString($schema, 'Schema should be returned as a string');

    // Top-level schema metadata.
    $this->assertArrayHasKey('type', $json_schema, 'Schema should have a type field');
    $this->assertEquals('object', $json_schema['type'], 'Schema should be of type object');

    // Check on fields.
    $this->assertArrayHasKey('properties', $json_schema, 'Schema should have properties');
    $properties = $json_schema['properties']['data']['properties']['attributes']['properties'];

    // There should be a title field in the schema.
    $this->assertArrayHasKey('title', $properties, 'Schema should have title field');
    $this->assertArrayHasKey('type', $properties['title'], 'Title should have type');
    $this->assertEquals('string', $properties['title']['type'], 'Title should be string type');
  }

  /**
   * Tests the normalizeResponse method.
   */
  public function testNormalizeResponse(): void {
    $bundle = 'article';

    // Test with json code block markers.
    $response = "```json\n{\"data\": {\"attributes\": {\"title\": \"Test Title\"}}}\n```";
    $result = $this->aiMigrator->normalizeResponse($response, $bundle);
    $this->assertIsArray($result, 'Response should be converted to array');
    $this->assertEquals('Test Title', $result['title'][0]['value'], 'JSON should be properly decoded');

    // Test with backticks.
    $response = "`{\"data\": {\"attributes\": {\"title\": \"Another Title\"}}}`";
    $result = $this->aiMigrator->normalizeResponse($response, $bundle);
    $this->assertIsArray($result, 'Response should be converted to array');
    $this->assertEquals('Another Title', $result['title'][0]['value'], 'JSON with backticks should be properly decoded');

    // Test with clean JSON.
    $response = '{"data": {"attributes": {"title": "Clean Title"}}}';
    $result = $this->aiMigrator->normalizeResponse($response, $bundle);
    $this->assertIsArray($result, 'Response should be converted to array');
    $this->assertEquals('Clean Title', $result['title'][0]['value'], 'Clean JSON should be properly decoded');

    // @todo Fix this when https://www.drupal.org/project/ai_migration/issues/3556037 is resolved.
    // Test with JSON with incorrect structure.
    // $response = '{"title": "Clean Title"}';
    // $result = $this->aiMigrator->normalizeResponse($response, $bundle);
    // $this->assertIsArray($result, 'Response should be an array');
    // $this->assertEquals(
    // [],
    // $result,
    // 'Improperly structured JSON should return empty array'
    // );
  }

}
