<?php

declare(strict_types=1);

namespace Drupal\Tests\ai_migration\Kernel;

use Drupal\ai_migration\Enum\PromptDefault;
use Drupal\ai_migration\Service\AiMigrationPromptManager;
use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\ai_migration\Trait\PromptRoleProviderTrait;
use PHPUnit\Framework\Attributes\CoversMethod;
use PHPUnit\Framework\Attributes\Group;

/**
 * @coversDefaultClass \Drupal\ai_migration\Service\AiMigrationPromptManager
 */
#[Group("ai_migration")]
#[CoversMethod(AiMigrationPromptManager::class, 'setConfig')]
#[CoversMethod(AiMigrationPromptManager::class, 'getPrompt')]
class PromptManagerTest extends KernelTestBase {

  use PromptRoleProviderTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'ai',
    'ai_migration',
    'file',
    'jsonapi',
    'schemata',
    'schemata_json_schema',
    'serialization',
  ];

  /**
   * The Prompt Plugin manager.
   *
   * @var \Drupal\ai_migration\Service\AiMigrationPromptManager
   */
  protected $promptManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->promptManager = $this->container->get('ai_migration.prompt_manager');
  }

  /**
   * Tests the getPrompt method for all roles.
   *
   * @dataProvider promptRoleProvider
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function testGetPromptWithPromptManager(string $role): void {
    $prompt = $this->randomString();

    $this->promptManager->setConfig([
      [
        'type' => 'string',
        'role' => $role,
        'prompt' => $prompt,
      ],
    ]);

    $result = $this->promptManager->getPrompt($role);
    $this->assertEquals($prompt, $result);
  }

  /**
   * Test the replace operation.
   *
   * @dataProvider promptRoleProvider
   */
  public function testPromptReplaceOperation(string $role): void {
    $prompt = $this->randomString();

    $config = [
      [
        'type' => 'string',
        'role' => $role,
        'prompt' => $prompt,
        'operation' => 'replace',
      ],
    ];

    $this->promptManager->setConfig($config);
    $result = $this->promptManager->getPrompt($role);

    $this->assertEquals($prompt, $result);
  }

  /**
   * Test the append operation.
   *
   * @dataProvider promptRoleProvider
   */
  public function testPromptAppendOperation(string $role): void {
    $default_prompt = PromptDefault::from($role)->getPromptText();
    $append = $this->randomString();
    $appended_string = $default_prompt . PHP_EOL . $append;

    $config = [
      [
        'type' => 'string',
        'role' => $role,
        'prompt' => $append,
        'operation' => 'append',
      ],
    ];

    $this->promptManager->setConfig($config);

    $result = $this->promptManager->getPrompt($role);
    $this->assertEquals($appended_string, $result);

    // Test appending with EOL set to FALSE.
    $config[0]['operation_eol'] = FALSE;
    $appended_string = $default_prompt . $append;

    $this->promptManager->setConfig($config);

    $result = $this->promptManager->getPrompt($role);
    $this->assertEquals($appended_string, $result);
  }

  /**
   * Test the append operation.
   *
   * @dataProvider promptRoleProvider
   */
  public function testPromptPrependOperation(string $role): void {
    $default_prompt = PromptDefault::from($role)->getPromptText();
    $prepend = $this->randomString();
    $prepend_string = $prepend . PHP_EOL . $default_prompt;

    $config = [
      [
        'type' => 'string',
        'role' => $role,
        'prompt' => $prepend,
        'operation' => 'prepend',
      ],
    ];

    $this->promptManager->setConfig($config);

    $result = $this->promptManager->getPrompt($role);
    $this->assertEquals($prepend_string, $result);

    // Test prepending with EOL set to FALSE.
    $config[0]['operation_eol'] = FALSE;
    $prepend_string = $prepend . $default_prompt;

    $this->promptManager->setConfig($config);

    $result = $this->promptManager->getPrompt($role);
    $this->assertEquals($prepend_string, $result);
  }

  /**
   * Tests that PluginException is thrown when type is not found for all roles.
   *
   * @dataProvider promptRoleProvider
   */
  public function testPluginExceptionWhenNoPluginFound(string $role): void {
    $this->promptManager->setConfig([
      [
        'type' => 'non_existent_type',
        'role' => $role,
      ],
    ]);

    $this->expectException(PluginException::class);
    $this->promptManager->getPrompt($role);
  }

  /**
   * Test InvalidArgumentException is thrown for an invalid operation.
   *
   * @dataProvider promptRoleProvider
   */
  public function testInvalidOperation(string $role): void {
    $this->promptManager->setConfig([
      [
        'type' => 'string',
        'role' => $role,
        'prompt' => 'test',
        'operation' => 'invalid_operation',
      ],
    ]);

    $this->expectException(\InvalidArgumentException::class);
    $this->promptManager->getPrompt($role);
  }

  /**
   * Test InvalidArgumentException is thrown for an invalid role.
   */
  public function testInvalidRole(): void {
    $role = 'invalid_role';

    $this->promptManager->setConfig([
      [
        'type' => 'string',
        'role' => $role,
        'prompt' => 'test',
      ],
    ]);

    $this->expectException(\InvalidArgumentException::class);
    $this->promptManager->getPrompt($role);
  }

}
