<?php

namespace Drupal\Tests\ai_migration\Unit;

use Drupal\ai_migration\HtmlSanitizerConfigBuilder;
use Drupal\ai_migration\Service\HtmlContentProcessor;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversMethod;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use Symfony\Component\HtmlSanitizer\HtmlSanitizer;

/**
 * @coversDefaultClass \Drupal\ai_migration\Service\HtmlContentProcessor
 */
#[Group("ai_migration")]
#[CoversMethod(HtmlContentProcessor::class, 'processHtml')]
#[CoversMethod(HtmlContentProcessor::class, 'extractFromContainer')]
#[CoversMethod(HtmlContentProcessor::class, 'setConfig')]
class HtmlContentProcessorTest extends UnitTestCase {

  /**
   * The logger factory mock.
   *
   * @var \PHPUnit\Framework\MockObject\MockObject
   */
  protected $loggerFactoryMock;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->loggerFactoryMock = $this->createMock(LoggerChannelFactoryInterface::class);
    $loggerChannel = $this->createMock(LoggerChannelInterface::class);
    $this->loggerFactoryMock->method('get')->willReturn($loggerChannel);
  }

  /**
   * Tests the processHtml method with sanitization enabled.
   *
   * @throws \PHPUnit\Framework\MockObject\Exception
   */
  public function testProcessHtmlWithSanitization() {
    $processor = new HtmlContentProcessor(
      $this->loggerFactoryMock
    );

    $html = '<div id="container" class="container" style="color: blue;" <a href="javascript:alert("XSS attack example");">Click me</a><script></script>Sanitized content</div>';

    $result = $processor->processHtml($html, [
      'allowSafeElements' => TRUE,
    ]);

    $this->assertEquals('<div id="container"><a>Click me</a>Sanitized content</div>', $result);

    // This also tests that the sanitizer object is reloaded when the config
    // is changed for processHtml.
    $result = $processor->processHtml($html, [
      'allowStaticElements' => TRUE,
    ]);

    $this->assertEquals('<div id="container" class="container" style="color: blue;"><a>Click me</a>Sanitized content</div>', $result);

    // Perform tests with the set methods for instances of injecting a
    // sanitizer interface object with a set config.
    $processor->setConfig([
      'allowSafeElements' => TRUE,
      'allowAttribute' => ['class', '*'],
    ]);
    $processor->setSanitizer(new HtmlSanitizer($processor->getConfig()));
    $result = $processor->processHtml($html);

    $this->assertEquals('<div id="container" class="container"><a>Click me</a>Sanitized content</div>', $result);
  }

  /**
   * Tests the container extraction functionality.
   */
  public function testContainerExtraction() {
    $processor = new HtmlContentProcessor($this->loggerFactoryMock);
    $html = '<html><div class="wrapper"><p>Target content</p></div></html>';

    $result = $processor->processHtml($html, [
      'container' => '.wrapper',
    ]);

    $this->assertEquals('<div class="wrapper"><p>Target content</p></div>', $result);
  }

  /**
   * Tests the setConfig method with invalid configurations.
   *
   * @dataProvider invalidConfigProvider
   */
  #[DataProvider('invalidConfigProvider')]
  public function testInvalidConfigThrowsException(array $invalidConfig, ?string $expectedExceptionMessage = NULL): void {
    $this->expectException(\InvalidArgumentException::class);
    if ($expectedExceptionMessage) {
      $this->expectExceptionMessage($expectedExceptionMessage);
    }

    (new HtmlSanitizerConfigBuilder)->build($invalidConfig);
  }

  /**
   * Data provider for invalid configuration test cases.
   */
  public static function invalidConfigProvider(): array {
    return [
      'conflicting options' => [
        [
          'allowSafeElements' => TRUE,
          'allowStaticElements' => TRUE,
        ],
        'Can only choose allowSafeElements OR allowStaticElements as a migrate config.',
      ],
      'unsupported option' => [
        [
          'head' => '',
        ],
        'Option "head" is not supported by html_processor.',
      ],
    ];
  }

}
