<?php

declare(strict_types = 1);

namespace Drupal\ai_more_like_this\Plugin\views\argument;

use Drupal\ai\Enum\VdbSimilarityMetrics;
use Drupal\ai_more_like_this\MoreLikeThisService;
use Drupal\ai_search\Plugin\search_api\backend\SearchApiAiSearchBackend;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\search_api\SearchApiException;
use Drupal\views\Plugin\views\argument\NumericArgument;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\views\Plugin\views\display\DisplayPluginBase;
use Drupal\views\ViewExecutable;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Form\FormStateInterface;
use Exception;

/**
 * Contextual filter that provides semantically similar nodes' IDs.
 *
 * @ViewsArgument("ai_more_like_this_entity_ids")
 */
class MoreLikeThisEntityIds extends NumericArgument implements ContainerFactoryPluginInterface {

  use LoggerChannelTrait;
  protected const string LOGGER_CHANNEL = 'ai_more_like_this';

  /**
   * Set to TRUE in setArgument().
   *
   * @var bool
   */
  protected bool $resolved = FALSE;

  /**
   * Initialization errors.
   *
   * @var array
   */
  private array $errors = [];

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    private readonly RouteMatchInterface $routeMatch,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly Connection $connection,
    private readonly MoreLikeThisService $moreLikeThis) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
    ) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_route_match'),
      $container->get('entity_type.manager'),
      $container->get('database'),
      $container->get('ai_more_like_this.mlt_service')
    );
  }

  /**
   * Overrides Drupal\views\Plugin\views\HandlerBase:init().
   *
   * {@inheritdoc}
   */
  public function init(ViewExecutable $view, DisplayPluginBase $display, ?array &$options = NULL) {
    parent::init($view, $display, $options);
    // Init MoreLikeThisService.
    try {
      $this->errors = $this->moreLikeThis->init($options);
    }
    catch (Exception $e) {
      $error_message = $this->t('Unable to initialize MoreLikeThisEntityIds service: @msg', [
        '@msg' => $e->getMessage()]);
      $this->getLogger(self::LOGGER_CHANNEL)->error($error_message);
      $this->errors[] = $error_message;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function defineOptions() {
    $options = parent::defineOptions();
    // Allow multiple.
    $options['break_phrase']['default'] = TRUE;
    // CSV format.
    $options['separator']['default'] = ',';
    // Force a harmless default setup so the hidden UI isn’t needed.
    $options['default_action']['default'] = 'default';
    $options['default_argument_type']['default'] = 'fixed';
    $options['default_argument_options']['default'] = ['argument' => ''];

    // (Optional) turn off validation UI, since we override validateArgument().
    $options['specify_validation']['default'] = FALSE;

    // RAG settings.
    $options['index'] = ['default' => NULL];
    $options['metric'] = ['default' => VdbSimilarityMetrics::CosineSimilarity->value];
    $options['distance_threshold'] = ['default' => 0.35];
    $options['embedding_strategy'] = ['default' => ''];

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    // 1) Hide/lock Views' default-argument UI we don't use.
    foreach ([
      'description', 'default_action', 'no_argument', 'exception',
      'default_argument_type', 'default_argument_options',
      'argument_present', 'token_help', 'specify_validation', 'validate',
    ] as $key) {
      if (isset($form[$key])) {
        $form[$key]['#access'] = FALSE;
      }
    }
    if (isset($form['break_phrase'])) {
      $form['break_phrase']['#default_value'] = TRUE;
      $form['break_phrase']['#access'] = FALSE;
    }
    if (isset($form['separator'])) {
      $form['separator']['#default_value'] = ',';
      $form['separator']['#access'] = FALSE;
    }

    // 1) Index selector — use Views UI temporary form flow so the dialog
    // does NOT close on first "Add & configure".
    $form['index'] = [
      '#type' => 'select',
      '#title' => $this->t('Source AI Search Index'),
      '#options' => $this->getAiSearchIndexes(),
      '#required' => TRUE,
      '#default_value' => $this->options['index'] ?? NULL,
      '#description' => $this->t('Index used to populate the Vector DB to be used for MLT.'),
      '#ajax' => [
        // Important: let Views UI handle the AJAX rebuild; no custom callback.
        'url' => views_ui_build_form_url($form_state),
        'event' => 'change',
      ],
      // Important: run the temporary submit
      // so Views rebuilds instead of closing.
      '#submit' => [[$this, 'submitTemporaryForm']],
      '#executes_submit_callback' => TRUE,
    ];

    // 2) Wrapper that will be replaced on AJAX rebuild.
    $form['rag_settings_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'rag-settings-wrapper'],
      '#attached' => [],
    ];

    // 3) On rebuild (after index change), values live under 'options'.
    $chosen_index = $form_state->getUserInput()['options']['index'] ?? $this->options['index'];
    if (!empty($chosen_index)) {
      // Build dependent fields.
      $form['rag_settings_wrapper']['rag_settings'] = $this->buildRagSettingsSubform($chosen_index, $form_state);
    }
  }

  /**
   * Build the dependent sub-form for a given index.
   *
   * @param string $chosen_index
   *   The Index machine name.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The FormState object.
   *
   * @return array
   *   The form array.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\search_api\SearchApiException
   */
  protected function buildRagSettingsSubform(string $chosen_index, FormStateInterface $form_state): array {
    $rag_settings = [
      '#type' => 'details',
      '#title' => $this->t('RAG Settings'),
      '#open' => TRUE,
      '#attached' => [],
    ];

    /** @var \Drupal\search_api\Entity\Index|null $index */
    $index = $this->entityTypeManager->getStorage('search_api_index')->load($chosen_index);
    if (!$index) {
      $rag_settings['msg'] = ['#markup' => $this->t('Selected index not found.')];
      return $rag_settings;
    }

    $backend = $index->getServerInstance()->getBackend();
    $backend_config = $index->getServerInstance()?->getBackendConfig();
    // Options must be scalars (no enums/objects in render arrays).
    $metric_options = [
      VdbSimilarityMetrics::CosineSimilarity->value  => $this->t('Cosine Similarity'),
      VdbSimilarityMetrics::EuclideanDistance->value => $this->t('Euclidean Distance'),
      VdbSimilarityMetrics::InnerProduct->value      => $this->t('Inner Product'),
    ];

    // Detect “index changed” on this AJAX rebuild.
    $prev_index = $this->options['index'] ?? NULL;
    $index_changed = ($chosen_index !== $prev_index);

    $metric_from_index = $backend_config['database_settings']['metric'];
    $metric_default = $index_changed
      ? $metric_from_index
      : ($this->options['metric'] ?? $metric_from_index);

    // Build the select.
    $rag_settings['metric'] = [
      '#type' => 'select',
      '#title' => $this->t('Similarity Metric'),
      '#options' => $metric_options,
      '#required' => TRUE,
      '#default_value' => $metric_default,
      '#description' => $this->t('Metric used for similarity calculations. By default the metric used in the AI Index selected, but it can be changed for better MLT results.'),
    ];
    // 👇 Critical: when the index changed, override the posted value once.
    if ($index_changed && $metric_from_index !== NULL) {
      $rag_settings['metric']['#value'] = (string) $metric_from_index;

      // Keep values consistent for the temp submit pipeline.
      $form_state->setValue('metric', (string) $metric_from_index);
      $opts = (array) $form_state->getValue('options', []);
      $opts['metric'] = (string) $metric_from_index;
      $form_state->setValue('options', $opts);

      // Also update raw user input so the rerender shows the new selection.
      $input = (array) $form_state->getUserInput();
      $input['options']['metric'] = (string) $metric_from_index;
      $form_state->setUserInput($input);
    }

    $rag_settings['distance_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Max distance'),
      '#description' => $this->t('Distance threshold (set for CosineSimilarity, for other metric type will be recalculated).'),
      '#default_value' => $this->options['distance_threshold'] ?? 0.35,
      '#min' => 0,
      '#max' => 1,
      '#step' => 0.01,
    ];

    // The Index Backend defines the strategy.
    $rag_settings['embedding_strategy'] = [
      '#type'  => 'select',
      '#title' => $this->t('Embeddings Strategy'),
      '#description' => $this->t('The Strategy is defined in AI Search Index Backend settings and cannot be changed.'),
      '#options' => $backend->getEmbeddingStrategiesOptions(),
      '#default_value' => $backend_config['embedding_strategy'],
      '#disabled' => TRUE,
    ];
    $this->options['embedding_strategy'] = $backend_config['embedding_strategy'];

    $rag_settings['metric']['#parents'] = ['metric'];
    $rag_settings['distance_threshold']['#parents'] = ['distance_threshold'];
    $rag_settings['embedding_strategy']['#parents'] = ['embedding_strategy'];

    return $rag_settings;
  }

  /**
   * {@inheritdoc}
   */
  public function submitOptionsForm(&$form, FormStateInterface $form_state): void {
    $option_values = &$form_state->getValue('options');
    if (empty($option_values)) {
      return;
    }
    $option_values['metric'] = $form_state->getValue('metric');
    $distance = (float) $form_state->getValue('distance_threshold');
    $option_values['distance_threshold'] = max(0.0, min(1.0, $distance));
    $option_values['embedding_strategy'] = $form_state->getValue('embedding_strategy');

  }

  /**
   * Gets all AI Search Indexes.
   *
   * @return array
   *   An array of valid Search AI indexes.
   */
  private function getAiSearchIndexes(): array {
    $ai_indexes = [];
    $indexes = $this->entityTypeManager->getStorage('search_api_index')
      ->loadMultiple();
    foreach ($indexes as $index) {
      // Only enabled indexes matter.
      if (!$index->status()) {
        continue;
      }
      try {
        // Indexes cannot be enabled without a connection
        // to a valid, enabled server.
        $backend = $index->getServerInstance()->getBackend();
        if ($backend instanceof SearchApiAiSearchBackend) {
          $ai_indexes[$index->id()] = $index->label() . ' (' . $index->id() . ')';
        }
      }
      catch (SearchApiException $e) {
        $this->getLogger(self::LOGGER_CHANNEL)->error('Error loading AI Search Server for the index @index: @msg', [
          '@index' => $index->id(),
          '@msg' => $e->getMessage(),
        ]);
      }

    }
    return $ai_indexes;
  }

  /**
   * {@inheritdoc}
   */
  public function setArgument($arg) {
    if ($this->resolved) {
      // Idempotent; Views may call this more than once.
      return TRUE;
    }
    $this->resolved = TRUE;
    // Find the "context" node ID (shown page), if any.
    // 1) If we're in Views UI "Preview with contextual filters" and a numeric
    // value was entered, use it as the context NID.
    $context_nid = NULL;
    if (!empty($this->view->live_preview) && is_scalar($arg) && preg_match('/^\s*\d+\s*$/', (string) $arg)) {
      $context_nid = (int) trim((string) $arg);
    }
    else {
      // 2) Otherwise resolve from the current route
      // (normal block/page runtime).
      $node = $this->routeMatch->getParameter('node');
      $context_nid = is_object($node) ? (int) $node->id() : (is_numeric($node) ? (int) $node : NULL);
    }
    if (!$context_nid) {
      return FALSE;
    }
    $limit = $this->getPagerLimit();
    // Ask the RAG service (vector DB) to provide similar NIDs.
    // Returns an array of integers; handle empty return FALSE.
    $nids = $this->moreLikeThis->semanticProximityNodeIds($context_nid, $limit);

    if (empty($nids)) {
      return FALSE;
    }

    $nids = array_values(array_unique(array_map('intval', (array) $nids)));
    // Populate BOTH properties that Views expects.
    $this->argument = implode(',', $nids);
    $this->value    = $nids;

    return TRUE;
  }

  /**
   * Build the WHERE ... nid IN (...) clause using vector search results.
   *
   * {@inheritdoc}
   */
  public function query($group_by = FALSE) {

    $alias_table = $this->query->ensureTable('node_field_data');
    // At this point $this->value has been rebuilt
    // from $this->argument as an array.
    $ids = array_map('intval', (array) $this->value);
    if (!$ids) {
      // No semantically close items -> no results.
      $this->query->addWhereExpression(0, '1=0');
      return;
    }
    $this->query->addWhere(0, "$alias_table.nid", $ids, 'IN');

    // Preserve the similarity order.
    $driver = $this->connection->driver();
    if ($driver === 'mysql' || $driver === 'mariadb') {
      // Keep the order exactly as in $this->value.
      $order_expr = "FIELD($alias_table.nid, " . implode(',', array_map('intval', $this->value)) . ")";
    } else {
      // Cross-DB fallback using CASE.
      $cases = [];
      foreach (array_values($this->value) as $pos => $nid) {
        $cases[] = 'WHEN ' . $alias_table . '.nid = ' . (int) $nid . ' THEN ' . ($pos + 1);
      }
      $order_expr = 'CASE ' . implode(' ', $cases) . ' ELSE 999999 END';
    }

    // Provide a clean alias, e.g. 'similarity_rank'.
    $this->query->addOrderBy(NULL, $order_expr, 'ASC', 'similarity_rank');
  }

  /**
   * Gets Pager Items Per Page.
   *
   * @return int
   *   Pager Items Per Page.
   */
  protected function getPagerLimit(): int {
    $limit = 0;

    $pager_conf = $this->view->display_handler->getOption('pager') ?? [];
    if (!empty($pager_conf['options']['items_per_page'])) {
      $limit = (int) $pager_conf['options']['items_per_page'];
    }

    return $limit;
  }

  /**
   * {@inheritdoc}
   */
  public function validate(): ?array {
    return $this->errors;
  }

  /**
   * Bypass default numeric validation.
   *
   *  {@inheritdoc}
   */
  public function validateArgument($arg) {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts() {
    return Cache::mergeContexts(parent::getCacheContexts(), ['route']);
  }

}
