<?php

namespace Drupal\ai_provider_acquia\Form;

use Drupal\ai_provider_acquia\Client\AcquiaAiClient;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Utility\Error;
use GuzzleHttp\Exception\GuzzleException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Acquia AI API access.
 */
final class AcquiaConfigForm extends ConfigFormBase {

  /**
   * Config settings.
   */
  const CONFIG_NAME = 'ai_provider_acquia.settings';

  /**
   * Default provider ID.
   */
  const PROVIDER_ID = 'acquia_ai_gateway';

  /**
   * The field plugin manager service.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  private $aiProviderManager;

  /**
   * Image style helper service.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  private $keyRepository;

  /**
   * Acquia DAM logger channel.
   *
   * @var \GuzzleHttp\Client
   */
  private $client;

  /**
   * {@inheritdoc}
   */
  final public static function create(ContainerInterface $container) {
    $instance = parent::create($container);

    $instance->aiProviderManager = $container->get('ai.provider');
    $instance->keyRepository = $container->get('key.repository');
    $instance->client = $container->get('http_client');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'acquiaai_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::CONFIG_NAME,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // The key name is hardcoded so other modules can reliably modify it.
    $api_key = $this->keyRepository->getKey('acquia_ai_gateway_key')?->getKeyValue() ?? NULL;
    $host = getenv('AI_GATEWAY_URL') ?? NULL;

    if (!$host || !$api_key) {
      $this->messenger()->addError("Unable to detect an API key or host. Please contact Acquia Support to ensure your application is enabled for AI capabilities.");
      return $form;
    }

    $config = $this->config(static::CONFIG_NAME);
    $form = parent::buildForm($form, $form_state);

    $form['host'] = [
      '#type' => 'markup',
      '#markup' => "Connected to host: " . $host,
    ];

    $form['moderation'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable moderation'),
      '#description' => $this->t('If OpenAI compatible moderation is not enabled, you should have another form of moderation enabled in Acquia AI.'),
      '#default_value' => (bool) $config->get('moderation'),
    ];

    $form['key_details'] = [
      '#type' => 'details',
      '#weight' => 1000,
      '#title' => $this->t('Key details'),
      '#open' => TRUE,
      'table' => [
        '#theme' => 'table',
        '#rows' => [],
      ],
    ];

    try {
      $client = new AcquiaAiClient($this->client, $host, $api_key);
      $keys = $client->keyInfo();
      $key_info = reset($keys);
      if ($key_info) {
        if ($key_info->info->key_alias) {
          $form['key_details']['table']['#rows'][] = [
            $this->t('Name'),
            $key_info->info->key_alias,
          ];
        }

        $form['key_details']['table']['#rows'][] = [
          $this->t('Key'),
          $key_info->info->key_name,
        ];

        $form['key_details']['table']['#rows'][] = [
          $this->t('Spend ($)'),
          number_format($key_info->info->spend, 5),
        ];

        $form['key_details']['table']['#rows'][] = [
          $this->t('Max budget ($)'),
          $key_info->info->max_budget === NULL ? $this->t('N/A') : number_format($key_info->info->max_budget, 2),
        ];

        $form['key_details']['table']['#rows'][] = [
          $this->t('Blocked'),
          $key_info->info->blocked ? $this->t('Yes') : $this->t('No'),
        ];

        foreach ($form['key_details']['table']['#rows'] as &$row) {
          $row[0] = [
            'data' => ['#markup' => $row[0]],
            'header' => TRUE,
          ];
        }
      }
    }
    catch (GuzzleException $e) {
      Error::logException(
        $this->logger('ai_provider_acquia'),
        $e,
      );
      $response = json_decode($e->getResponse()->getBody()->getContents());
      if ($e->getCode() === 400 && $response?->error?->type === 'budget_exceeded') {
        $this->messenger()->addError($response?->error?->message ?? $this->t('You have exceeded your budget.'));
      }
      elseif ($e->getCode() === 401) {
        $this->messenger()->addError($this->t('Invalid API key.'));
      }
      else {
        $this->messenger()->addError($this->t('Unable to retrieve key information.'));
      }
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Set default models.
    $this->setDefaultModels();
    parent::submitForm($form, $form_state);
  }

  /**
   * Set default models for the AI provider.
   */
  private function setDefaultModels() {
    // Create provider instance.
    $provider = $this->aiProviderManager->createInstance(static::PROVIDER_ID);

    // Check if getSetupData() method exists and is callable.
    if (is_callable([$provider, 'getSetupData'])) {
      // Fetch setup data.
      $setup_data = $provider->getSetupData();

      // Ensure the setup data is valid.
      if (!empty($setup_data) && is_array($setup_data) && !empty($setup_data['default_models']) && is_array($setup_data['default_models'])) {
        // Loop through and set default models for each operation type.
        foreach ($setup_data['default_models'] as $op_type => $model_id) {
          $this->aiProviderManager->defaultIfNone($op_type, static::PROVIDER_ID, $model_id);
        }
      }
    }
  }

}
