<?php

namespace Drupal\ai_provider_acquia\Plugin\AiProvider;

use Drupal\ai\Base\OpenAiBasedProviderClientBase;
use Drupal\ai\Exception\AiSetupFailureException;
use Drupal\ai\Traits\OperationType\EmbeddingsTrait;
use Drupal\ai_provider_acquia\Client\AcquiaAiClient;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\AiProvider;
use Drupal\ai\OperationType\Chat\ChatInterface;
use Drupal\ai\OperationType\Embeddings\EmbeddingsInterface;
use Drupal\ai\OperationType\Moderation\ModerationInterface;
use Drupal\ai\OperationType\SpeechToText\SpeechToTextInterface;
use Drupal\ai\OperationType\TextToImage\TextToImageInterface;
use Drupal\ai\OperationType\TextToSpeech\TextToSpeechInterface;
use Drupal\ai\Traits\OperationType\ChatTrait;
use OpenAI\Client;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'acquia' LiteLLM provider.
 */
#[AiProvider(
  id: 'acquia_ai_gateway',
  label: new TranslatableMarkup('Acquia'),
)]
class AcquiaAiProvider extends OpenAiBasedProviderClientBase implements
  ContainerFactoryPluginInterface,
  ChatInterface,
  ModerationInterface,
  EmbeddingsInterface,
  TextToSpeechInterface,
  SpeechToTextInterface,
  TextToImageInterface {

  use ChatTrait;
  use EmbeddingsTrait;

  /**
   * The AI cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $aiCache;

  /**
   * The Acquia AI API client.
   *
   * @var \Drupal\ai_provider_acquia\Client\AcquiaAiClient
   */
  protected $acquiaAiClient;

  /**
   * API Key.
   *
   * @var string
   */
  protected string $apiKey = '';

  /**
   * Run moderation call, before a normal call.
   *
   * @var bool|null
   */
  protected bool|null $moderation = NULL;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $parent_instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $parent_instance->aiCache = $container->get('cache.ai');

    return $parent_instance;
  }

  /**
   * {@inheritdoc}
   */
  public function hasAuthentication(): bool {
    // Parent methods directly call config->get if authentication isn't set.
    if (empty($this->apiKey) && $api_key = $this->keyRepository->getKey('acquia_ai_gateway_key')?->getKeyValue())  {
        $this->apiKey = $api_key;
    }
    return !empty($this->apiKey);
  }

  /**
   * {@inheritdoc}
   */
  protected function loadClient(): void {
    $config = $this->getConfig();
    $host = !empty($config->get('host')) ? $config->get('host') : getenv('AI_GATEWAY_URL') ?? NULL;
    $api_key = $this->apiKey ?? $this->keyRepository->getKey('acquia_ai_gateway_key')?->getKeyValue() ?? '';

    $this->setAuthentication($api_key);
    $this->acquiaAiClient = new AcquiaAiClient(
      $this->httpClient,
      $host,
      $this->apiKey,
    );

    // Set custom endpoint from host config if available.
    if (!empty($host)) {
      $this->setEndpoint($host);
    }

    try {
      parent::loadClient();
    }
    catch (AiSetupFailureException $e) {
      throw new AiSetupFailureException('Failed to initialize Acquia AI client: ' . $e->getMessage(), $e->getCode(), $e);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguredModels(?string $operation_type = NULL, array $capabilities = []): array {
    // Load all models, and since OpenAI does not provide information about
    // which models does what, we need to hard code it in a helper function.
    $this->loadClient();
    return $this->getModels($operation_type ?? '', $capabilities);
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedOperationTypes(): array {
    return [
      'audio_to_audio',
      'chat',
      'embeddings',
      'moderation',
      'text_to_image',
      'text_to_speech',
      'image_and_audio_to_video',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getModelSettings(string $model_id, array $generalConfig = []): array {
    $this->loadClient();
    $model_info = $this->acquiaAiClient->models()[$model_id] ?? NULL;

    if (!$model_info) {
      return $generalConfig;
    }

    foreach (array_keys($generalConfig) as $name) {
      if (!in_array($name, $model_info->supportedOpenAiParams)) {
        unset($generalConfig[$name]);
      }
    }

    return $generalConfig;
  }

  /**
   * Gets the raw client.
   *
   * @param string $api_key
   *   If the API key should be hot swapped.
   *
   * @return \OpenAI\Client
   *   The OpenAI Compatible client.
   */
  public function getClient(string $api_key = ''): Client {
    // If the moderation is not set, we load it from the configuration.
    if (is_null($this->moderation)) {
      $this->moderation = $this->getConfig()->get('moderation');
    }
    if ($api_key) {
      $this->setAuthentication($api_key);
    }
    $this->loadClient();
    return $this->client;
  }

  /**
   * Retrieves and filters a list of models from the Acquia AI client.
   *
   * Filters out deprecated or unsupported models based on the operation type.
   * The Acquia AI API does not natively filter these models.
   *
   * @param string $operation_type
   *   The bundle to filter models by.
   * @param array $capabilities
   *   The capabilities to filter models by.
   *
   * @return array
   *   A filtered list of public models.
   */
  public function getModels(string $operation_type, array $capabilities): array {
    $models = [];
    foreach ($this->acquiaAiClient->models() as $model) {
      switch ($operation_type) {
        case 'text_to_image':
          if ($model->supportsImageOutput) {
            $models[$model->name] = $model->name;
          }
          break;

        case 'text_to_speech':
          if ($model->supportsAudioOutput) {
            $models[$model->name] = $model->name;
          }
          break;

        case 'audio_to_audio':
          if ($model->supportsAudioInput && $model->supportsAudioOutput) {
            $models[$model->name] = $model->name;
          }
          break;

        case 'moderation':
          if ($model->supportsModeration) {
            $models[$model->name] = $model->name;
          }
          break;

        case 'embeddings':
          if ($model->supportsEmbeddings) {
            $models[$model->name] = $model->name;
          }
          break;

        case 'chat':
          if ($model->supportsChat) {
            $models[$model->name] = $model->name;
          }
          break;

        case 'image_and_audio_to_video':
          if ($model->supportsImageAndAudioToVideo) {
            $models[$model->name] = $model->name;
          }
          break;

        default:
          break;
      }
    }
    return $models;
  }

}
