<?php

namespace Drupal\ai_provider_alibabacloud;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;

/**
 * Helper service for Alibaba Cloud Model Studio operations.
 */
class AlibabaCloudHelper {

  use StringTranslationTrait;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected MessengerInterface $messenger;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * Constructs a new AlibabaCloudHelper object.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service.
   */
  public function __construct(
    MessengerInterface $messenger,
    ClientInterface $http_client,
    LoggerChannelFactoryInterface $logger_factory,
  ) {
    $this->messenger = $messenger;
    $this->httpClient = $http_client;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * Gets the base URL for the API based on mode and region.
   *
   * @param string $mode
   *   The API mode ('compatible' or 'native').
   * @param string $region
   *   The region ('intl' or 'cn').
   *
   * @return string
   *   The base URL.
   */
  public function getBaseUrl(string $mode, string $region): string {
    $urls = [
      'compatible' => [
        'intl' => 'https://dashscope-intl.aliyuncs.com/compatible-mode/v1',
        'cn' => 'https://dashscope.aliyuncs.com/compatible-mode/v1',
      ],
      'native' => [
        'intl' => 'https://dashscope-intl.aliyuncs.com/api/v1',
        'cn' => 'https://dashscope.aliyuncs.com/api/v1',
      ],
    ];

    return $urls[$mode][$region] ?? $urls['compatible']['intl'];
  }

  /**
   * Tests the API connection with the provided configuration.
   *
   * @param string $api_key
   *   The API key to test.
   * @param string $mode
   *   The API mode.
   * @param string $region
   *   The region.
   *
   * @return bool
   *   TRUE if the connection is successful, FALSE otherwise.
   */
  public function testConnection(string $api_key, string $mode, string $region): bool {
    $base_url = $this->getBaseUrl($mode, $region);

    try {
      if ($mode === 'compatible') {
        $url = $base_url . '/chat/completions';
        $payload = [
          'model' => 'qwen-turbo',
          'messages' => [
            ['role' => 'user', 'content' => 'Test'],
          ],
          'max_tokens' => 10,
        ];
      }
      else {
        $url = $base_url . '/services/aigc/text-generation/generation';
        $payload = [
          'model' => 'qwen-turbo',
          'input' => [
            'messages' => [
              ['role' => 'user', 'content' => 'Test'],
            ],
          ],
          'parameters' => [
            'max_tokens' => 10,
          ],
        ];
      }

      $response = $this->httpClient->request('POST', $url, [
        'headers' => [
          'Authorization' => 'Bearer ' . $api_key,
          'Content-Type' => 'application/json',
        ],
        'json' => $payload,
        'timeout' => 10,
      ]);

      $body = Json::decode($response->getBody()->getContents());

      // Check for valid response structure.
      if ($mode === 'compatible') {
        return isset($body['choices']);
      }
      else {
        return isset($body['output']);
      }
    }
    catch (RequestException $e) {
      $logger = $this->loggerFactory->get('ai_provider_alibabacloud');
      $logger->error('API connection test failed: @message', ['@message' => $e->getMessage()]);

      if ($e->getCode() === 401) {
        $this->messenger->addError($this->t('Invalid API key. Please check your Alibaba Cloud Model Studio API key.'));
      }
      elseif ($e->getCode() === 403) {
        $this->messenger->addError($this->t('Access denied. Please verify your API key permissions.'));
      }
      else {
        $this->messenger->addError($this->t('Connection failed: @message', ['@message' => $e->getMessage()]));
      }

      return FALSE;
    }
  }

  /**
   * Gets available models for the configured region.
   *
   * @return array
   *   Array of available models.
   */
  public function getAvailableModels(): array {
    // Alibaba Cloud doesn't provide a models list endpoint.
    // Return predefined models based on documentation.
    return [
      'qwen-max' => 'Qwen Max (Most Powerful)',
      'qwen-plus' => 'Qwen Plus (Balanced)',
      'qwen-turbo' => 'Qwen Turbo (Fast)',
      'qwen-flash' => 'Qwen Flash (Simple & Fast)',
      'qwen-coder' => 'Qwen Coder (Code Generation)',
      'text-embedding-v1' => 'Text Embedding v1',
      'text-embedding-v2' => 'Text Embedding v2',
      'text-embedding-v3' => 'Text Embedding v3',
      'text-embedding-v4' => 'Text Embedding v4 (Latest)',
    ];
  }

}
