<?php

namespace Drupal\ai_provider_alibabacloud;

use Drupal\ai\OperationType\Chat\StreamedChatMessage;
use Drupal\ai\OperationType\Chat\StreamedChatMessageIterator;

/**
 * Iterator for processing streamed chat messages from Alibaba Cloud.
 */
class AlibabaCloudStreamIterator extends StreamedChatMessageIterator {

  /**
   * The API mode being used.
   *
   * @var string
   */
  protected string $apiMode;

  /**
   * Constructs a new AlibabaCloudStreamIterator.
   *
   * @param mixed $iterator
   *   The stream iterator.
   * @param string $api_mode
   *   The API mode ('compatible' or 'native').
   */
  public function __construct($iterator, string $api_mode = 'compatible') {
    parent::__construct($iterator);
    $this->apiMode = $api_mode;
  }

  /**
   * {@inheritdoc}
   */
  public function getIterator(): \Generator {
    $buffer = '';

    foreach ($this->iterator as $chunk) {
      $buffer .= $chunk;

      // Normalize line endings for consistent processing.
      $buffer = str_replace("\r\n", "\n", $buffer);

      // Process complete SSE events from buffer.
      while (($pos = strpos($buffer, "\n\n")) !== FALSE) {
        $event = substr($buffer, 0, $pos);
        $buffer = substr($buffer, $pos + 2);

        if (strpos($event, 'data: ') === 0) {
          $data = substr($event, 6);
          $data = trim($data);

          if ($data === '[DONE]') {
            return;
          }

          $decoded = json_decode($data, TRUE);
          if ($decoded) {
            yield $this->parseMessage($decoded);
          }
        }
      }
    }
  }

  /**
   * Parses a message based on the API mode.
   *
   * @param array $data
   *   The decoded JSON data.
   *
   * @return \Drupal\ai\OperationType\Chat\StreamedChatMessage
   *   The parsed streamed chat message.
   */
  protected function parseMessage(array $data): StreamedChatMessage {
    if ($this->apiMode === 'compatible') {
      // OpenAI-compatible format.
      $content = $data['choices'][0]['delta']['content'] ?? '';
      $role = $data['choices'][0]['delta']['role'] ?? 'assistant';
      $usage = $data['usage'] ?? [];
    }
    else {
      // Native DashScope format with incremental output.
      $content = $data['output']['text'] ?? '';
      $role = 'assistant';
      $usage = $data['usage'] ?? [];
    }

    return new StreamedChatMessage($role, $content, $usage);
  }

}
