<?php

namespace Drupal\ai_provider_alibabacloud\Plugin\AiProvider;

use Drupal\ai\Attribute\AiProvider;
use Drupal\ai\Base\AiProviderClientBase;
use Drupal\ai\Enum\AiProviderCapability;
use Drupal\ai\Exception\AiQuotaException;
use Drupal\ai\Exception\AiRateLimitException;
use Drupal\ai\Exception\AiResponseErrorException;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatInterface;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\Chat\ChatOutput;
use Drupal\ai\OperationType\Chat\Tools\ToolsFunctionOutput;
use Drupal\ai\OperationType\Embeddings\EmbeddingsInput;
use Drupal\ai\OperationType\Embeddings\EmbeddingsInterface;
use Drupal\ai\OperationType\Embeddings\EmbeddingsOutput;
use Drupal\ai\Traits\OperationType\ChatTrait;
use Drupal\ai_provider_alibabacloud\AlibabaCloudHelper;
use Drupal\ai_provider_alibabacloud\AlibabaCloudStreamIterator;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Psr7\StreamWrapper;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the Alibaba Cloud Model Studio provider.
 */
#[AiProvider(
  id: 'alibabacloud',
  label: new TranslatableMarkup('Alibaba Cloud Model Studio'),
)]
class AlibabaCloudProvider extends AiProviderClientBase implements
  ContainerFactoryPluginInterface,
  ChatInterface,
  EmbeddingsInterface {

  use ChatTrait;

  /**
   * The Alibaba Cloud helper service.
   *
   * @var \Drupal\ai_provider_alibabacloud\AlibabaCloudHelper
   */
  protected AlibabaCloudHelper $alibabaCloudHelper;

  /**
   * The API key.
   *
   * @var string
   */
  protected string $apiKey = '';

  /**
   * The API mode.
   *
   * @var string
   */
  protected string $apiMode = 'compatible';

  /**
   * The region.
   *
   * @var string
   */
  protected string $region = 'intl';

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->alibabaCloudHelper = $container->get('ai_provider_alibabacloud.helper');
    $instance->logger = $container->get('logger.factory')->get('ai_provider_alibabacloud');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguredModels(?string $operation_type = NULL, array $capabilities = []): array {
    $models = [];

    switch ($operation_type) {
      case 'chat':
        $models = [
          'qwen-max' => 'Qwen Max',
          'qwen-plus' => 'Qwen Plus',
          'qwen-turbo' => 'Qwen Turbo',
          'qwen-flash' => 'Qwen Flash',
          'qwen-coder' => 'Qwen Coder',
        ];
        break;

      case 'embeddings':
        $models = [
          'text-embedding-v1' => 'Text Embedding v1',
          'text-embedding-v2' => 'Text Embedding v2',
          'text-embedding-v3' => 'Text Embedding v3',
          'text-embedding-v4' => 'Text Embedding v4',
        ];
        break;
    }

    return $models;
  }

  /**
   * {@inheritdoc}
   */
  public function isUsable(?string $operation_type = NULL, array $capabilities = []): bool {
    // Check if API key is configured.
    $config = $this->getConfig();
    if (!$config->get('key_id')) {
      return FALSE;
    }

    // Check if operation type is supported.
    if ($operation_type) {
      return in_array($operation_type, $this->getSupportedOperationTypes());
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedOperationTypes(): array {
    return [
      'chat',
      'embeddings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedCapabilities(): array {
    return [
      AiProviderCapability::StreamChatOutput,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig(): ImmutableConfig {
    return $this->configFactory->get('ai_provider_alibabacloud.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function getApiDefinition(): array {
    $definition_file = $this->moduleHandler->getModule('ai_provider_alibabacloud')->getPath() . '/definitions/api_defaults.yml';
    return Yaml::parseFile($definition_file);
  }

  /**
   * {@inheritdoc}
   */
  public function getModelSettings(string $model_id, array $generalConfig = []): array {
    // Adjust max tokens based on model.
    switch ($model_id) {
      case 'qwen-turbo':
      case 'qwen-flash':
        $generalConfig['max_tokens']['default'] = 2000;
        $generalConfig['max_tokens']['constraints']['max'] = 8000;
        break;

      case 'qwen-plus':
      case 'qwen-max':
      case 'qwen-coder':
        $generalConfig['max_tokens']['default'] = 4000;
        $generalConfig['max_tokens']['constraints']['max'] = 32768;
        break;
    }

    // Add model-specific parameters based on API mode.
    if ($this->apiMode === 'native') {
      // Native mode specific parameters.
      $generalConfig['repetition_penalty']['default'] = 1.0;
      $generalConfig['enable_thinking']['default'] = FALSE;
      $generalConfig['thinking_budget']['default'] = 0;
    }

    return $generalConfig;
  }

  /**
   * {@inheritdoc}
   */
  public function setAuthentication(mixed $authentication): void {
    $this->apiKey = $authentication;
  }

  /**
   * {@inheritdoc}
   */
  public function chat(array|string|ChatInput $input, string $model_id, array $tags = []): ChatOutput {
    $this->loadApiConfiguration();

    // Normalize input to messages array.
    $messages = $this->normalizeChatInput($input);

    // Build request payload based on API mode.
    if ($this->apiMode === 'compatible') {
      $payload = $this->buildCompatiblePayload($messages, $model_id);
      $url = $this->alibabaCloudHelper->getBaseUrl('compatible', $this->region) . '/chat/completions';
    }
    else {
      $payload = $this->buildNativePayload($messages, $model_id);
      $url = $this->alibabaCloudHelper->getBaseUrl('native', $this->region) . '/services/aigc/text-generation/generation';
    }

    // Add tools if present.
    if ($input instanceof ChatInput && $input->getChatTools()) {
      $this->addToolsToPayload($payload, $input);
    }

    // Add structured response if present.
    if ($input instanceof ChatInput && $input->getChatStructuredJsonSchema()) {
      $this->addStructuredResponseToPayload($payload, $input);
    }

    try {
      if ($this->streamed) {
        return $this->streamChat($url, $payload, $input);
      }
      else {
        return $this->regularChat($url, $payload, $input);
      }
    }
    catch (RequestException $e) {
      $this->handleApiException($e);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function embeddings(string|EmbeddingsInput $input, string $model_id, array $tags = []): EmbeddingsOutput {
    $this->loadApiConfiguration();

    // Normalize input.
    if ($input instanceof EmbeddingsInput) {
      $input = $input->getPrompt();
    }

    // Only compatible mode supports embeddings.
    $url = $this->alibabaCloudHelper->getBaseUrl('compatible', $this->region) . '/embeddings';
    $payload = [
      'model' => $model_id,
      'input' => $input,
    ];

    try {
      $response = $this->httpClient->request('POST', $url, [
        'headers' => [
          'Authorization' => 'Bearer ' . $this->apiKey,
          'Content-Type' => 'application/json',
        ],
        'json' => $payload,
        'timeout' => $this->getConfig()->get('timeout') ?: 60,
      ]);

      $body = Json::decode($response->getBody()->getContents());

      if (!isset($body['data'][0]['embedding'])) {
        throw new AiResponseErrorException('Invalid embeddings response from Alibaba Cloud API.');
      }

      return new EmbeddingsOutput($body['data'][0]['embedding'], $body, []);
    }
    catch (RequestException $e) {
      $this->handleApiException($e);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function embeddingsVectorSize(string $model_id): int {
    return match($model_id) {
      'text-embedding-v1' => 1536,
      'text-embedding-v2' => 1536,
      'text-embedding-v3' => 1024,
      'text-embedding-v4' => 1024,
      default => 0,
    };
  }

  /**
   * {@inheritdoc}
   */
  public function maxEmbeddingsInput($model_id = ''): int {
    // Alibaba Cloud text embedding models support up to 2048 tokens.
    return 2048;
  }

  /**
   * {@inheritdoc}
   */
  public function getSetupData(): array {
    return [
      'key_config_name' => 'key_id',
      'default_models' => [
        'chat' => 'qwen-plus',
        'chat_with_tools' => 'qwen-max',
        'chat_with_structured_response' => 'qwen-max',
        'embeddings' => 'text-embedding-v4',
      ],
    ];
  }

  /**
   * Loads API configuration from settings.
   */
  protected function loadApiConfiguration(): void {
    $config = $this->getConfig();

    // Load API key from Key module.
    if (!$this->apiKey) {
      $key_id = $config->get('key_id');
      if ($key_id && $this->keyRepository) {
        $key = $this->keyRepository->getKey($key_id);
        if ($key) {
          $this->apiKey = $key->getKeyValue();
        }
      }
    }

    $this->apiMode = $config->get('api_mode') ?: 'compatible';
    $this->region = $config->get('region') ?: 'intl';
  }

  /**
   * Normalizes chat input to messages array format.
   *
   * @param array|string|\Drupal\ai\OperationType\Chat\ChatInput $input
   *   The input to normalize.
   *
   * @return array
   *   Normalized messages array.
   */
  protected function normalizeChatInput(array|string|ChatInput $input): array {
    $messages = [];

    if (is_string($input)) {
      $messages[] = ['role' => 'user', 'content' => $input];
    }
    elseif (is_array($input)) {
      $messages = $input;
    }
    elseif ($input instanceof ChatInput) {
      // Add system role if configured.
      if ($this->chatSystemRole) {
        $messages[] = [
          'role' => 'system',
          'content' => $this->chatSystemRole,
        ];
      }

      // Process each message.
      foreach ($input->getMessages() as $message) {
        $content = [
          [
            'type' => 'text',
            'text' => $message->getText(),
          ],
        ];

        // Add images if present.
        if (count($message->getImages())) {
          foreach ($message->getImages() as $image) {
            $content[] = [
              'type' => 'image_url',
              'image_url' => [
                'url' => $image->getAsBase64EncodedString(),
              ],
            ];
          }
        }

        $new_message = [
          'role' => $message->getRole(),
          'content' => $content,
        ];

        // Handle tool messages.
        if ($message->getToolsId()) {
          $new_message['tool_call_id'] = $message->getToolsId();
        }

        if ($message->getTools()) {
          $new_message['tool_calls'] = $message->getRenderedTools();
        }

        $messages[] = $new_message;
      }
    }

    return $messages;
  }

  /**
   * Builds payload for OpenAI-compatible mode.
   *
   * @param array $messages
   *   The messages array.
   * @param string $model_id
   *   The model ID.
   *
   * @return array
   *   The payload array.
   */
  protected function buildCompatiblePayload(array $messages, string $model_id): array {
    $payload = [
      'model' => $model_id,
      'messages' => $messages,
    ];

    // Add configuration parameters.
    if (isset($this->configuration['temperature'])) {
      $payload['temperature'] = $this->configuration['temperature'];
    }
    if (isset($this->configuration['top_p'])) {
      $payload['top_p'] = $this->configuration['top_p'];
    }
    if (isset($this->configuration['presence_penalty'])) {
      $payload['presence_penalty'] = $this->configuration['presence_penalty'];
    }
    if (isset($this->configuration['frequency_penalty'])) {
      $payload['frequency_penalty'] = $this->configuration['frequency_penalty'];
    }
    if (isset($this->configuration['max_tokens'])) {
      $payload['max_tokens'] = $this->configuration['max_tokens'];
    }

    return $payload;
  }

  /**
   * Builds payload for DashScope native mode.
   *
   * @param array $messages
   *   The messages array.
   * @param string $model_id
   *   The model ID.
   *
   * @return array
   *   The payload array.
   */
  protected function buildNativePayload(array $messages, string $model_id): array {
    $payload = [
      'model' => $model_id,
      'input' => [
        'messages' => $messages,
      ],
      'parameters' => [],
    ];

    // Map configuration to parameters.
    if (isset($this->configuration['temperature'])) {
      $payload['parameters']['temperature'] = $this->configuration['temperature'];
    }
    if (isset($this->configuration['top_p'])) {
      $payload['parameters']['top_p'] = $this->configuration['top_p'];
    }
    if (isset($this->configuration['repetition_penalty'])) {
      $payload['parameters']['repetition_penalty'] = $this->configuration['repetition_penalty'];
    }
    if (isset($this->configuration['enable_thinking'])) {
      $payload['parameters']['enable_thinking'] = $this->configuration['enable_thinking'];
    }
    if (isset($this->configuration['thinking_budget'])) {
      $payload['parameters']['thinking_budget'] = $this->configuration['thinking_budget'];
    }
    if (isset($this->configuration['max_tokens'])) {
      $payload['parameters']['max_tokens'] = $this->configuration['max_tokens'];
    }

    return $payload;
  }

  /**
   * Adds tools to the payload.
   *
   * @param array &$payload
   *   The payload array.
   * @param \Drupal\ai\OperationType\Chat\ChatInput $input
   *   The chat input.
   */
  protected function addToolsToPayload(array &$payload, ChatInput $input): void {
    $tools = $input->getChatTools()->renderToolsArray();

    if ($this->apiMode === 'compatible') {
      $payload['tools'] = $tools;
    }
    else {
      $payload['parameters']['tools'] = $tools;
    }
  }

  /**
   * Adds structured response schema to the payload.
   *
   * @param array &$payload
   *   The payload array.
   * @param \Drupal\ai\OperationType\Chat\ChatInput $input
   *   The chat input.
   */
  protected function addStructuredResponseToPayload(array &$payload, ChatInput $input): void {
    $schema = [
      'type' => 'json_schema',
      'json_schema' => $input->getChatStructuredJsonSchema(),
    ];

    if ($this->apiMode === 'compatible') {
      $payload['response_format'] = $schema;
    }
    else {
      $payload['parameters']['response_format'] = $schema;
    }
  }

  /**
   * Performs a regular (non-streaming) chat request.
   *
   * @param string $url
   *   The API endpoint URL.
   * @param array $payload
   *   The request payload.
   * @param mixed $input
   *   The original input.
   *
   * @return \Drupal\ai\OperationType\Chat\ChatOutput
   *   The chat output.
   */
  protected function regularChat(string $url, array $payload, $input): ChatOutput {
    $response = $this->httpClient->request('POST', $url, [
      'headers' => [
        'Authorization' => 'Bearer ' . $this->apiKey,
        'Content-Type' => 'application/json',
      ],
      'json' => $payload,
      'timeout' => $this->getConfig()->get('timeout') ?: 60,
    ]);

    $body = Json::decode($response->getBody()->getContents());

    if ($this->apiMode === 'compatible') {
      // Parse OpenAI-compatible response.
      if (!isset($body['choices'][0]['message'])) {
        throw new AiResponseErrorException('Invalid response structure from Alibaba Cloud API.');
      }

      $message_data = $body['choices'][0]['message'];
      $content = $message_data['content'] ?? '';
      $role = $message_data['role'] ?? 'assistant';

      // Handle tools if present.
      $tools = [];
      if (!empty($message_data['tool_calls']) && $input instanceof ChatInput && $input->getChatTools()) {
        foreach ($message_data['tool_calls'] as $tool_call) {
          $arguments = Json::decode($tool_call['function']['arguments']);
          $tools[] = new ToolsFunctionOutput(
            $input->getChatTools()->getFunctionByName($tool_call['function']['name']),
            $tool_call['id'],
            $arguments
          );
        }
      }
    }
    else {
      // Parse native response.
      if (!isset($body['output']['text'])) {
        throw new AiResponseErrorException('Invalid response structure from Alibaba Cloud API.');
      }

      $content = $body['output']['text'];
      $role = 'assistant';
      $tools = [];
    }

    $message = new ChatMessage($role, $content, []);
    if (!empty($tools)) {
      $message->setTools($tools);
    }

    return new ChatOutput($message, $body, []);
  }

  /**
   * Performs a streaming chat request.
   *
   * @param string $url
   *   The API endpoint URL.
   * @param array $payload
   *   The request payload.
   * @param mixed $input
   *   The original input.
   *
   * @return \Drupal\ai\OperationType\Chat\ChatOutput
   *   The chat output with streaming iterator.
   */
  protected function streamChat(string $url, array $payload, $input): ChatOutput {
    $config = $this->getConfig();

    // Set streaming parameters.
    if ($this->apiMode === 'compatible') {
      $payload['stream'] = TRUE;

      // Add stream_options if configured.
      if ($config->get('include_usage_in_stream')) {
        $payload['stream_options'] = ['include_usage' => TRUE];
      }

      $headers = [
        'Authorization' => 'Bearer ' . $this->apiKey,
        'Content-Type' => 'application/json',
      ];
    }
    else {
      $payload['parameters']['incremental_output'] = TRUE;
      $headers = [
        'Authorization' => 'Bearer ' . $this->apiKey,
        'Content-Type' => 'application/json',
        'X-DashScope-SSE' => 'enable',
      ];
    }

    $response = $this->httpClient->request('POST', $url, [
      'headers' => $headers,
      'json' => $payload,
      'stream' => TRUE,
      'timeout' => $config->get('timeout') ?: 60,
    ]);

    $stream = StreamWrapper::getResource($response->getBody());
    $iterator = new AlibabaCloudStreamIterator($stream, $this->apiMode);

    return new ChatOutput($iterator, [], []);
  }

  /**
   * Handles API exceptions and converts them to appropriate AI exceptions.
   *
   * @param \GuzzleHttp\Exception\RequestException $e
   *   The request exception to handle.
   *
   * @throws \Drupal\ai\Exception\AiQuotaException
   *   When quota is exceeded.
   * @throws \Drupal\ai\Exception\AiRateLimitException
   *   When rate limit is exceeded.
   * @throws \Exception
   *   For other errors.
   */
  protected function handleApiException(RequestException $e): void {
    $response_body = '';
    if ($e->hasResponse()) {
      $response_body = $e->getResponse()->getBody()->getContents();
    }

    // Check for rate limit errors.
    if ($e->getCode() === 429 || strpos($response_body, 'rate_limit') !== FALSE) {
      throw new AiRateLimitException('Rate limit exceeded for Alibaba Cloud API.');
    }

    // Check for quota errors.
    if (strpos($response_body, 'quota') !== FALSE || strpos($response_body, 'insufficient_balance') !== FALSE) {
      throw new AiQuotaException('API quota exceeded or insufficient balance.');
    }

    // Log the error and re-throw.
    $this->logger->error('Alibaba Cloud API error: @message', ['@message' => $e->getMessage()]);
    throw $e;
  }

}
