# Alibaba Cloud Model Studio AI Provider

The **Alibaba Cloud Model Studio AI Provider** module integrates
Alibaba Cloud's Qwen models and DashScope API services into Drupal,
enabling powerful AI capabilities through the Drupal AI module ecosystem.
This provider bridges your Drupal applications with Alibaba Cloud's
cutting-edge language models and embedding services.

## Features

- **Dual API Mode Support**: Choose between OpenAI-compatible mode for easy
migration or DashScope native mode for advanced features
- **Multiple Model Access**: Leverage the full Qwen model family
(Max, Plus, Turbo, Flash, Coder) for various use cases
- **Advanced Embeddings**: Support for text embedding models
v1-v4 with configurable dimensions
- **Regional Flexibility**: Connect to Singapore (International)
or Beijing (China) regions based on your requirements
- **Streaming Responses**: Real-time streaming support for
interactive chat experiences
- **Secure Key Management**: Integration with Drupal's Key module
for secure API credential storage
- **Comprehensive Configuration**: Fine-tune model parameters including
temperature, tokens, penalties, and more
- **Enterprise Ready**: Production-grade integration with timeout controls,
error handling, and usage tracking

## Supported Models

### Chat Models
- **Qwen Max**: Most powerful model for complex reasoning and creative tasks
- **Qwen Plus**: Balanced performance for general-purpose applications
- **Qwen Turbo**: Fast responses for real-time interactions
- **Qwen Flash**: Ultra-fast model for simple tasks
- **Qwen Coder**: Specialized model for code generation and analysis

### Embedding Models
- **Text Embedding v1-v3**: Legacy embedding models with 1536/1024 dimensions
- **Text Embedding v4**: Latest embedding model with enhanced performance

## Requirements

- **Drupal Core**: ^10.3 or ^11
- **AI Module**: The core AI module must be installed and enabled
- **Key Module**: ^1.18 for secure API key management
- **Alibaba Cloud Account**: Valid API key from Model Studio

## Installation

1. **Install via Composer**:

   composer require drupal/ai_provider_alibabacloud

2. **Enable the Module**:

   drush en ai_provider_alibabacloud

3. **Configure API Key**:
  - Navigate to `/admin/config/system/keys/add`
  - Create a new key entity for your Alibaba Cloud API key
  - Enter your API key from [Model Studio Singapore](https://modelstudio-intl.console.aliyun.com/)
or [Model Studio Beijing](https://bailian.console.aliyun.com/)

4. **Configure the Provider**:
  - Go to `/admin/config/ai/providers/alibabacloud`
  - Select your API key entity
  - Choose your preferred region and API mode
  - Adjust advanced settings as needed

## Configuration Options

### Basic Settings
- **API Key**: Select the Key entity containing your Model Studio API key
- **Region**: Choose between Singapore (International) or Beijing (China)
- **API Mode**: Select OpenAI-compatible or DashScope native mode

### Advanced Settings
- **Streaming**: Enable/disable Server-Sent Events for streaming responses
- **Timeout**: Configure request timeout (10-300 seconds)
- **Usage Tracking**: Include token usage in streaming responses

### Model Parameters
- **Temperature**: Control randomness (0-2)
- **Max Tokens**: Limit response length (up to 32768 for Qwen Max/Plus)
- **Top P**: Nucleus sampling probability (0-1)
- **Penalties**: Adjust presence, frequency, or repetition penalties
- **Thinking Mode**: Enable Qwen3 thinking mode (native mode only)

## Usage Examples

Once configured, the Alibaba Cloud provider integrates seamlessly with
Drupal's AI module features:

### Content Generation
Generate articles, summaries, and creative content using Qwen's powerful
language models.

### Code Development
Leverage Qwen Coder for generating, reviewing, and documenting code
directly within Drupal.

### Multilingual Support
Translate and localize content with Qwen's multilingual capabilities.

### Semantic Search
Use text embeddings to build intelligent search and recommendation systems.

### Smart Content Moderation
Automatically review and moderate user-generated content using AI analysis.

### Interactive Chatbots
Build conversational interfaces with streaming support for real-time
interactions.

## API Modes Comparison

### OpenAI-Compatible Mode
- Drop-in replacement for OpenAI API
- Easy migration from other providers
- Standard request/response format
- Best for existing OpenAI integrations

### DashScope Native Mode
- Access to Alibaba Cloud exclusive features
- Thinking mode for enhanced reasoning
- Incremental output streaming
- Optimized for Qwen models

## Development

### Programmatic Usage

Use the AI module's service to interact with Alibaba Cloud models:

    $ai_provider = \Drupal::service('ai.provider');
    $provider = $ai_provider->createInstance('alibabacloud');

    // Chat completion
    $input = new ChatInput([
      new ChatMessage('user', 'Hello, how can you help me?')
    ]);
    $response = $provider->chat($input, 'qwen-plus');

    // Generate embeddings
    $embedding_input = new EmbeddingsInput('Text to embed');
    $embeddings = $provider->embeddings($embedding_input, 'text-embedding-v4');

### Custom Integration

Extend the provider for custom implementations:

    class MyCustomProvider extends AlibabaCloudProvider {
      // Custom logic here
    }

## Troubleshooting

### Common Issues

**Invalid API Key Error**
- Verify your API key is correct and active
- Check the selected region matches your API key's region
- Ensure the Key entity has the correct value

**Connection Timeout**
- Increase the timeout value in advanced settings
- Check your server's network connectivity to Alibaba Cloud endpoints
- Verify firewall rules allow HTTPS connections

**Model Not Available**
- Confirm the selected model is available in your region
- Check your Alibaba Cloud account has access to the model
- Try switching between API modes

### Debug Mode

Enable debug logging to troubleshoot issues:

    drush config:set ai_provider_alibabacloud.settings debug_mode 1

Check logs at `/admin/reports/dblog` filtered by type
"ai_provider_alibabacloud".

## Performance Optimization

### Caching
The module automatically caches model responses when appropriate.
Configure cache settings at `/admin/config/development/performance`.

### Streaming
Enable streaming for better user experience with long responses:
- Reduces time to first token
- Provides real-time feedback
- Lower memory usage for large responses

### Model Selection
Choose the right model for your use case:
- **Qwen Flash**: Simple queries, fastest response
- **Qwen Turbo**: Balance of speed and quality
- **Qwen Plus**: Complex tasks, high quality
- **Qwen Max**: Maximum capability, slower response

## Security Considerations

- **API Key Storage**: Always use the Key module for storing API credentials
- **Rate Limiting**: Configure appropriate rate limits to prevent abuse
- **Input Validation**: Sanitize user inputs before sending to the API
- **Region Compliance**: Choose the appropriate region for data sovereignty
requirements

## Contributing

Contributions are welcome! Please submit issues and merge requests on the
[Drupal.org project page](https://www.drupal.org/project/ai_provider_alibabacloud).

### Development Setup

    git clone https://git.drupalcode.org/project/ai_provider_alibabacloud.git
    cd ai_provider_alibabacloud
    composer install

### Testing

Run the test suite:

    phpunit tests/
    phpcs --standard=Drupal,DrupalPractice .
    phpstan analyse

## Resources

- [Module Page](https://www.drupal.org/project/ai_provider_alibabacloud)
- [Issue Queue](https://www.drupal.org/project/issues/ai_provider_alibabacloud)
- [Alibaba Cloud Model Studio Documentation](https://help.aliyun.com/product/2400256.html)
- [Drupal AI Module](https://www.drupal.org/project/ai)
- [DashScope API Reference](https://dashscope.aliyun.com/api)

## License

This project is licensed under the GNU General Public License v2.0 or later.

## Maintainers

Current maintainer:
- **Mahyar SBT (mahyarsbt)** -
[https://www.drupal.org/u/mahyarsbt](https://www.drupal.org/u/mahyarsbt)

## Credits

Special thanks to the Drupal AI module maintainers and the Alibaba Cloud team
for their excellent documentation and support.
