<?php

namespace Drupal\ai_provider_alibabacloud\Form;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai_provider_alibabacloud\AlibabaCloudHelper;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\key\KeyRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for Alibaba Cloud Model Studio AI Provider.
 */
class AlibabaCloudConfigForm extends ConfigFormBase {

  /**
   * The configuration name.
   */
  const CONFIG_NAME = 'ai_provider_alibabacloud.settings';

  /**
   * The provider ID.
   */
  const PROVIDER_ID = 'alibabacloud';

  /**
   * The AI provider plugin manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected AiProviderPluginManager $aiProviderManager;

  /**
   * The key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * The Alibaba Cloud helper service.
   *
   * @var \Drupal\ai_provider_alibabacloud\AlibabaCloudHelper
   */
  protected AlibabaCloudHelper $alibabaCloudHelper;

  /**
   * Constructs a new AlibabaCloudConfigForm object.
   *
   * @param \Drupal\ai\AiProviderPluginManager $ai_provider_manager
   *   The AI provider plugin manager.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The key repository.
   * @param \Drupal\ai_provider_alibabacloud\AlibabaCloudHelper $alibaba_cloud_helper
   *   The Alibaba Cloud helper service.
   */
  public function __construct(
    AiProviderPluginManager $ai_provider_manager,
    KeyRepositoryInterface $key_repository,
    AlibabaCloudHelper $alibaba_cloud_helper,
  ) {
    $this->aiProviderManager = $ai_provider_manager;
    $this->keyRepository = $key_repository;
    $this->alibabaCloudHelper = $alibaba_cloud_helper;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('ai.provider'),
      $container->get('key.repository'),
      $container->get('ai_provider_alibabacloud.helper')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_provider_alibabacloud_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [static::CONFIG_NAME];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(static::CONFIG_NAME);

    $form['key_id'] = [
      '#type' => 'key_select',
      '#title' => $this->t('Alibaba Cloud Model Studio API Key'),
      '#description' => $this->t('Select the Key entity containing your API key. Obtain an API key from the <a href=":intl" target="_blank">Singapore region</a> or <a href=":cn" target="_blank">Beijing region</a>.', [
        ':intl' => 'https://modelstudio-intl.console.aliyun.com/',
        ':cn' => 'https://bailian.console.aliyun.com/',
      ]),
      '#default_value' => $config->get('key_id'),
      '#required' => TRUE,
    ];

    $form['region'] = [
      '#type' => 'select',
      '#title' => $this->t('Region'),
      '#description' => $this->t('Select the region for your API endpoint.'),
      '#options' => [
        'intl' => $this->t('Singapore (International)'),
        'cn' => $this->t('Beijing (China)'),
      ],
      '#default_value' => $config->get('region') ?: 'intl',
      '#required' => TRUE,
    ];

    $form['api_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('API Mode'),
      '#description' => $this->t('Choose between OpenAI-compatible and DashScope native modes.'),
      '#options' => [
        'compatible' => $this->t('OpenAI Compatible'),
        'native' => $this->t('DashScope Native'),
      ],
      '#default_value' => $config->get('api_mode') ?: 'compatible',
      '#required' => TRUE,
    ];

    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced Settings'),
      '#open' => FALSE,
    ];

    $form['advanced']['enable_streaming'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable streaming responses'),
      '#description' => $this->t('Enable Server-Sent Events (SSE) for streaming chat responses.'),
      '#default_value' => $config->get('enable_streaming') ?: FALSE,
    ];

    $form['advanced']['include_usage_in_stream'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include usage info in streaming'),
      '#description' => $this->t('Include token usage information in streaming responses (compatible mode only).'),
      '#default_value' => $config->get('include_usage_in_stream') ?: FALSE,
      '#states' => [
        'visible' => [
          ':input[name="enable_streaming"]' => ['checked' => TRUE],
          ':input[name="api_mode"]' => ['value' => 'compatible'],
        ],
      ],
    ];

    $form['advanced']['timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Request timeout'),
      '#description' => $this->t('Maximum time in seconds to wait for API responses.'),
      '#default_value' => $config->get('timeout') ?: 60,
      '#min' => 10,
      '#max' => 300,
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $key_id = $form_state->getValue('key_id');
    $region = $form_state->getValue('region');
    $mode = $form_state->getValue('api_mode');

    if (empty($key_id)) {
      $form_state->setErrorByName('key_id', $this->t('The API key is required.'));
      return;
    }

    $key = $this->keyRepository->getKey($key_id);
    if (!$key) {
      $form_state->setErrorByName('key_id', $this->t('The selected key does not exist.'));
      return;
    }

    $api_key = $key->getKeyValue();
    if (empty($api_key)) {
      $form_state->setErrorByName('key_id', $this->t('The selected key value is empty.'));
      return;
    }

    // Test the API connection.
    if (!$this->alibabaCloudHelper->testConnection($api_key, $mode, $region)) {
      $form_state->setErrorByName('key_id', $this->t('Failed to connect to Alibaba Cloud Model Studio API. Please check your configuration.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config(static::CONFIG_NAME)
      ->set('key_id', $form_state->getValue('key_id'))
      ->set('region', $form_state->getValue('region'))
      ->set('api_mode', $form_state->getValue('api_mode'))
      ->set('enable_streaming', $form_state->getValue('enable_streaming'))
      ->set('include_usage_in_stream', $form_state->getValue('include_usage_in_stream'))
      ->set('timeout', $form_state->getValue('timeout'))
      ->save();

    // Set default models for the provider.
    $this->setDefaultModels();

    parent::submitForm($form, $form_state);
  }

  /**
   * Sets default models for various operation types.
   */
  protected function setDefaultModels(): void {
    $provider = $this->aiProviderManager->createInstance(static::PROVIDER_ID);

    if (is_callable([$provider, 'getSetupData'])) {
      $setup_data = $provider->getSetupData();

      if (!empty($setup_data['default_models'])) {
        foreach ($setup_data['default_models'] as $op_type => $model_id) {
          $this->aiProviderManager->defaultIfNone($op_type, static::PROVIDER_ID, $model_id);
        }
      }
    }
  }

}
