<?php

namespace Drupal\ai_provider_apertus\Form;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\Service\AiProviderFormHelper;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Apertus AI API access.
 */
class ApertusConfigForm extends ConfigFormBase {

  /**
   * Config settings.
   */
  const CONFIG_NAME = 'ai_provider_apertus.settings';

  /**
   * Default provider ID.
   */
  const PROVIDER_ID = 'apertus';

  /**
   * The AI provider manager.
   *
   * @var AiProviderPluginManager
   */
  protected $aiProviderManager;
  
   /**
   * The form helper.
   *
   * @var AiProviderFormHelper
   */
  protected $formHelper;

  /**
   * Constructs a new ApertusConfigForm object.
   */
  final public function __construct(AiProviderPluginManager $ai_provider_manager, AiProviderFormHelper $form_helper) {
    $this->aiProviderManager = $ai_provider_manager;
    $this->formHelper = $form_helper;
  }

  /**
   * {@inheritdoc}
   */
  final public static function create(ContainerInterface $container) {
    return new static(
      $container->get('ai.provider'),
      $container->get('ai.form_helper')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'apertus_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::CONFIG_NAME,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(static::CONFIG_NAME);

    $form['api_key'] = [
      '#type' => 'key_select',
      '#title' => $this->t('Apertus AI API Key'),
      '#description' => $this->t('The API Key. Create key here: <a href="https://platform.publicai.co/settings/api-keys">https://platform.publicai.co/settings/api-keys</a>.'),
      '#default_value' => $config->get('api_key'),
    ];
    
    $form['load_models_table'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Load models table'),
      '#description' => $this->t('To make this work you may need to apply patch from <a target="blank" href="@url">[#3545188]</a>',
        ['@url' => 'https://www.drupal.org/project/ai/issues/3545188']
       ),
      '#default_value' => 0,
      '#ajax' => [
        'callback' => '::loadModelsTable',
        'wrapper' => 'models__table',
      ],
    ];
    
    $form['models_table'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'models__table'
      ]
    ];
    
    return parent::buildForm($form, $form_state);
  }
  
  /**
   * AjaxCallback to load the models table
   * 
   * @param array $form
   * @param FormStateInterface $form_state
   * @return type
   */
  public function loadModelsTable(array &$form, FormStateInterface $form_state) {
    $provider = $this->aiProviderManager->createInstance(static::PROVIDER_ID);
    return $this->formHelper->getModelsTable($form, $form_state, $provider);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Retrieve the configuration.
    $this->config(static::CONFIG_NAME)
      ->set('api_key', $form_state->getValue('api_key'))
      ->save();

    // Set default models.
    $this->setDefaultModels();

    parent::submitForm($form, $form_state);
  }

  /**
   * Set default models for the AI provider.
   */
  private function setDefaultModels() {
    // Create provider instance.
    $provider = $this->aiProviderManager->createInstance(static::PROVIDER_ID);

    // Check if getSetupData() method exists and is callable.
    if (is_callable([$provider, 'getSetupData'])) {
      // Fetch setup data.
      $setup_data = $provider->getSetupData();

      // Ensure the setup data is valid.
      if (!empty($setup_data) && is_array($setup_data) && !empty($setup_data['default_models']) && is_array($setup_data['default_models'])) {
        // Loop through and set default models for each operation type.
        foreach ($setup_data['default_models'] as $op_type => $model_id) {
          $this->aiProviderManager->defaultIfNone($op_type, static::PROVIDER_ID, $model_id);
        }
      }
    }
  }

}
