<?php

namespace Drupal\ai_provider_apertus\Plugin\AiProvider;

use Drupal\Component\Serialization\Json;
use Drupal\ai\Attribute\AiProvider;
use Drupal\ai\Base\OpenAiBasedProviderClientBase;
use Drupal\ai\Exception\AiQuotaException;
use Drupal\ai\Exception\AiRateLimitException;
use Drupal\ai\Exception\AiSetupFailureException;
use Drupal\ai\Traits\OperationType\ChatTrait;
use Drupal\Component\Utility\Crypt;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the 'apertus' provider.
 */
#[AiProvider(
  id: 'apertus',
  label: new TranslatableMarkup('Apertus AI')
)]
class ApertusProvider extends OpenAiBasedProviderClientBase {

  use ChatTrait;

  /**
   * {@inheritdoc}
   */
  protected string $endpoint = 'https://api.publicai.co/v1';

  /**
   * {@inheritdoc}
   */
  public function getConfiguredModels(?string $operation_type = NULL, array $capabilities = []): array {

    $this->loadClient();
    return $this->getModels($operation_type ?? '', $capabilities);

  }
  
  public function getModels(string $operation_type, $capabilities): array {
    $models = [];

    $cache_key = 'apertus_models_' . $operation_type . '_' . Crypt::hashBase64(Json::encode($capabilities));
    $cache_data = $this->cacheBackend->get($cache_key);

    if (!empty($cache_data)) {
      return $cache_data->data;
    }

    $list = $this->client->models()->list()->toArray();

    foreach ($list['data'] as $model) {      
      $models[$model['id']] = $model['id'];
    }

    if (!empty($models)) {
      asort($models);
      $this->cacheBackend->set($cache_key, $models);
    }

    return $models;
  }

  public function isUsable(?string $operation_type = NULL, array $capabilities = []): bool {
      return in_array($operation_type, $this->getSupportedOperationTypes());
  }
  
  /**
   * {@inheritdoc}
   */
  public function getSupportedOperationTypes(): array {
    return [
      'chat',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getModelSettings(string $model_id, array $generalConfig = []): array {
    return $generalConfig;
  }
  
  /**
   * {@inheritdoc}
   */
  protected function loadClient(): void {
    // Set custom endpoint from host config if available.
    if (!empty($this->getConfig()->get('host'))) {
      $this->setEndpoint($this->getConfig()->get('host'));
    }

    try {
      parent::loadClient();
    }
    catch (AiSetupFailureException $e) {
      throw new AiSetupFailureException('Failed to initialize Apertus client: ' . $e->getMessage(), $e->getCode(), $e);
    }
  }

  public function getSetupData(): array {
    return [
      'key_config_name' => 'api_key',
      'default_models' => [
      ],
    ];
  }


  /**
   * Handle API exceptions consistently.
   *
   * @param Exception $e
   *   The exception to handle.
   *
   * @throws AiRateLimitException
   * @throws AiQuotaException
   * @throws Exception
   */
  protected function handleApiException(\Exception $e): void {
    if (strpos($e->getMessage(), 'Too Many Requests') !== FALSE) {
      throw new AiRateLimitException($e->getMessage());
    }
    throw $e;
  }
  
   /**
   * {@inheritdoc}
   */
  public function loadModelsForm(array $form, $form_state, string $operation_type, string|NULL $model_id = NULL): array {
    // maybe extend later
    return parent::loadModelsForm($form, $form_state, $operation_type, str_replace('--slash--', '/', $model_id));
  }

  public function loadModelConfig(string $operation_type, string|NULL $model_id): array {
    return parent::loadModelConfig($operation_type, str_replace('--slash--', '/', $model_id));
  }
  
  public function modelSupportsCapabilities(string $operation_type, string $model_id, array $capabilities): bool {
    return $operation_type === 'chat' && in_array(\Drupal\ai\Enum\AiModelCapability::ChatJsonOutput, $capabilities);
  }

}
