<?php

declare(strict_types=1);

namespace Drupal\ai_provider_aws_bedrock;

use Drupal\ai_provider_aws_bedrock\Attribute\BedrockModel;
use Drupal\ai_provider_aws_bedrock\Models\BedrockModelInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Bedrock Model plugin manager.
 */
final class BedrockModelPluginManager extends DefaultPluginManager {

  use StringTranslationTrait;

  /**
   * Constructs the object.
   */
  public function __construct(
    \Traversable $namespaces,
    CacheBackendInterface $cache_backend,
    ModuleHandlerInterface $module_handler,
  ) {
    parent::__construct('Models', $namespaces, $module_handler, BedrockModelInterface::class, BedrockModel::class);
    $this->alterInfo('bedrock_model_info');
    $this->setCacheBackend($cache_backend, 'bedrock_model_plugins');
  }

  /**
   * Create an instance of a Bedrock Model plugin based on a model ID.
   *
   * Attempts to find a plugin definition where the model ID starts with the
   * definition ID.
   */
  public function createInstanceFromModelId(string $model_id, array $configuration = []): ?BedrockModelInterface {
    $definitions = $this->getDefinitions();
    $definitions = array_filter($definitions, static fn($definition) => str_starts_with($model_id, $definition['id']));
    if (!$definitions) {
      return NULL;
    }
    // In theory, we could get multiple matches. For now lets return the first.
    // An improvement could be to return the one with the longest match.
    $definition = reset($definitions);
    try {
      return $this->createInstance($definition['id'], $configuration);
    }
    catch (\Exception) {
      return NULL;
    }
  }

}
