<?php

namespace Drupal\ai_provider_aws_bedrock\Models\TextToImage;

use Drupal\ai_provider_aws_bedrock\Attribute\BedrockModel;
use Drupal\ai\Exception\AiBrokenOutputException;
use Drupal\ai\OperationType\GenericType\ImageFile;
use Drupal\ai_provider_aws_bedrock\Models\BedrockModelInterface;
use Drupal\ai_provider_aws_bedrock\Models\ModelWithInputOutputInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * The Stable Diffusion model.
 */
#[BedrockModel(
  id: 'stability.stable-diffusion-xl',
  label: new TranslatableMarkup('Stable Diffusion'),
)]
class StableDiffusion implements BedrockModelInterface, ModelWithInputOutputInterface {

  /**
   * {@inheritdoc}
   */
  public static function providerConfig(&$config, $model_id): void {
    $config['resolution'] = [
      'type' => 'string',
      'constraints' => [
        'options' => [
          '1024x1024',
          '1152x896',
          '1216x832',
          '1536x640',
          '640x1536',
          '832x1216',
          '896x1152',
        ],
      ],
      'label' => t('Resolution'),
    ];

    $config['cfg_scale'] = [
      'type' => 'number',
      'default' => 7,
      'label' => t('CFG Scale'),
      'constraints' => [
        'min' => 0,
        'max' => 35,
      ],
    ];

    $config['steps'] = [
      'type' => 'number',
      'default' => 30,
      'label' => t('Steps'),
      'constraints' => [
        'min' => 10,
        'max' => 150,
      ],
    ];

    $config['style_preset'] = [
      'type' => 'string',
      'constraints' => [
        'options' => [
          '3d-model',
          'analog-film',
          'anime',
          'cinematic',
          'comic-book',
          'digital-art',
          'enhance',
          'fantasy-art',
          'isometric',
          'line-art',
          'low-poly',
          'modeling-compound',
          'neon-punk',
          'origami',
          'photographic',
          'pixel-art',
          'tile-texture',
        ],
      ],
      'label' => t('Style Preset'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function formatInput($input = "", $image = NULL, $config = [], $model_id = ''): array {
    $payload = [
      'text_prompts' => [
        ['text' => $input],
      ],
    ];
    foreach ($config as $key => $value) {
      if ($key == 'resolution' && $value) {
        $parts = explode('x', $value);
        $payload['width'] = (int) $parts[0];
        $payload['height'] = (int) $parts[1];
      }
      elseif ($value) {
        $payload[$key] = $key == 'style_preset' ? $value : (int) $value;
      }
    }
    return $payload;
  }

  /**
   * {@inheritdoc}
   */
  public static function formatOutput($output, $config = []): array {
    if (empty($output['artifacts'][0]['base64'])) {
      throw new AiBrokenOutputException('No image data found in the response.');
    }
    $images = [];
    foreach ($output['artifacts'] as $image) {
      $images[] = new ImageFile(base64_decode($image['base64']), 'image/png', 'bedrock.png');
    }
    return $images;
  }

  /**
   * {@inheritdoc}
   */
  public static function providerCapabilities($capabilities, $model_id): bool {
    return TRUE;
  }

}
