<?php

namespace Drupal\ai_provider_aws_bedrock\Form;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure AWS Bedrock API access.
 */
class BedrockConfigForm extends ConfigFormBase {

  /**
   * Config settings.
   */
  const CONFIG_NAME = 'ai_provider_aws_bedrock.settings';

  /**
   * Constructs a new AWS Bedrock Config object.
   */
  final public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected CacheBackendInterface $cacheBackend,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  final public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('cache.default'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'bedrock_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::CONFIG_NAME,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(static::CONFIG_NAME);

    $profiles = $this->getAllProfiles();
    if (empty($profiles)) {
      $form['no_profiles'] = [
        '#markup' => '<p>' . $this->t('No profiles found. Please create a profile first in the %link settings page.', [
          '%link' => Link::createFromRoute($this->t('Amazon Web Services'), 'aws.overview')->toString(),
        ]) . '</p>',
      ];
      return $form;
    }

    $form['profile'] = [
      '#type' => 'select',
      '#title' => $this->t('AWS Bedrock Profile'),
      '#options' => $profiles,
      '#required' => TRUE,
      '#empty_option' => $this->t('-- Select a profile --'),
      '#description' => $this->t('The AWS Profile to use by default. Can be setup in the %link settings page.', [
        '%link' => Link::createFromRoute($this->t('Amazon Web Services'), 'aws.overview')->toString(),
      ]
      ),
      '#default_value' => $config->get('profile'),
    ];

    $form['on_demand'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Only show On-Demand Models'),
      '#description' => $this->t('If checked, only models that are on-demand will be shown. If you need provisioned models, uncheck this.'),
      '#default_value' => $config->get('on_demand'),
    ];

    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced'),
      '#open' => FALSE,
    ];

    $form['advanced']['chat_manual_models'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Chat Manual Models'),
      '#description' => $this->t('Since AWS offers granular access control, in theory an user can have access to run models, but not list models. In that case you can add chat models here. This is a new line separated list of models to use, with their id. This will append to the default model list.'),
      '#default_value' => $config->get('chat_manual_models'),
      '#rows' => 5,
    ];

    $form['advanced']['embeddings_manual_models'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Embeddings Manual Models'),
      '#description' => $this->t('Since AWS offers granular access control, in theory an user can have access to run models, but not list models. In that case you can add embeddings models here. This is a new line separated list of models to use, with their id. This will append to the default model list.'),
      '#default_value' => $config->get('embeddings_manual_models'),
      '#rows' => 5,
    ];

    $form['advanced']['text_to_image_manual_models'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Text to Image Manual Models'),
      '#description' => $this->t('Since AWS offers granular access control, in theory an user can have access to run models, but not list models. In that case you can add text to image models here. This is a new line separated list of models to use, with their id. This will append to the default model list.'),
      '#default_value' => $config->get('text_to_image_manual_models'),
      '#rows' => 5,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Retrieve the configuration.
    $this->config(static::CONFIG_NAME)
      ->set('profile', $form_state->getValue('profile'))
      ->set('on_demand', $form_state->getValue('on_demand'))
      ->set('chat_manual_models', $form_state->getValue('chat_manual_models'))
      ->set('embeddings_manual_models', $form_state->getValue('embeddings_manual_models'))
      ->set('text_to_image_manual_models', $form_state->getValue('text_to_image_manual_models'))
      ->save();
    // Clear the cache.
    if ($this->cacheBackend instanceof CacheTagsInvalidatorInterface) {
      // If the cache backend supports cache tags, invalidate the tag.
      $this->cacheBackend->invalidateTags(['aws_bedrock_models']);
    }
    else {
      // Otherwise, delete all cache items.
      $this->cacheBackend->deleteAll();
    }

    parent::submitForm($form, $form_state);
  }

  /**
   * Get all profiles.
   *
   * @return array
   *   An array of profiles.
   */
  protected function getAllProfiles() {
    $profiles = [];
    foreach ($this->entityTypeManager->getStorage('aws_profile')->loadMultiple() as $profile) {
      $profiles[$profile->id()] = $profile->label();
    }
    return $profiles;
  }

}
