<?php

namespace Drupal\ai_provider_aws_bedrock\Models\Embeddings;

use Drupal\ai_provider_aws_bedrock\Attribute\BedrockModel;
use Drupal\ai\Exception\AiBadRequestException;
use Drupal\ai\Exception\AiBrokenOutputException;
use Drupal\ai_provider_aws_bedrock\Models\BedrockModelInterface;
use Drupal\ai_provider_aws_bedrock\Models\ModelWithInputOutputInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * The Cohere Embeddings model.
 */
#[BedrockModel(
  id: 'cohere.embed',
  label: new TranslatableMarkup('Cohere Embeddings'),
)]
class CohereEmbeddings implements BedrockModelInterface, ModelWithInputOutputInterface {

  /**
   * {@inheritdoc}
   */
  public static function providerConfig(&$config, $model_id): void {
    // cohere.embed-english-v3
    // cohere.embed-multilingual-v3
    // This has to exist for RAG to work, even if its fixed.
    $config['dimensions'] = [
      'type' => 'number',
      'constraints' => [
        'options' => [
          '1024',
        ],
      ],
      'label' => t('Dimensions'),
      'default' => 1024,
      'required' => TRUE,
    ];

    $config['input_type'] = [
      'type' => 'string',
      'constraints' => [
        'options' => [
          'search_document',
          'search_query',
          'classification',
          'clustering',
        ],
      ],
      'label' => t('Input type'),
      'default' => 'search_query',
      'required' => TRUE,
    ];

    $config['truncate'] = [
      'type' => 'string',
      'constraints' => [
        'options' => [
          'NONE',
          'START',
          'END',
        ],
      ],
      'label' => t('Truncate'),
    ];

    $config['embedding_types'] = [
      'type' => 'string',
      'constraints' => [
        'options' => [
          'float',
          'int8',
          'uint8',
          'binary',
          'ubinary',
        ],
      ],
      'label' => t('Embedding Types'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function formatInput($input = "", $image = NULL, $config = [], $model_id = ''): array {
    // This should actually not be sent as a parameter.
    if (isset($config['dimensions'])) {
      unset($config['dimensions']);
    }

    foreach ($config as $key => $val) {
      if (empty($val)) {
        unset($config[$key]);
      }
    }

    if ($input) {
      $config['texts'] = [
        $input,
      ];
    }
    else {
      throw new AiBadRequestException('You need to give an input text.');
    }
    // Set default input type.
    $config['input_type'] = 'search_document';

    return $config;
  }

  /**
   * {@inheritdoc}
   */
  public static function formatOutput($output, $config = []): array {
    if (empty($output['embeddings'][0])) {
      throw new AiBrokenOutputException('No image data found in the response.');
    }
    return $output['embeddings'][0];
  }

  /**
   * {@inheritdoc}
   */
  public static function providerCapabilities($capabilities, $model_id): bool {
    return TRUE;
  }

}
