<?php

namespace Drupal\ai_provider_aws_bedrock\Models\Embeddings;

use Drupal\ai_provider_aws_bedrock\Attribute\BedrockModel;
use Drupal\ai\Exception\AiBadRequestException;
use Drupal\ai\Exception\AiBrokenOutputException;
use Drupal\ai_provider_aws_bedrock\Models\BedrockModelInterface;
use Drupal\ai_provider_aws_bedrock\Models\ModelWithInputOutputInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * The Titan embeddings model.
 */
#[BedrockModel(
  id: 'amazon.titan-embed',
  label: new TranslatableMarkup('Titan embeddings'),
)]
class TitanEmbeddings implements BedrockModelInterface, ModelWithInputOutputInterface {

  /**
   * {@inheritdoc}
   */
  public static function providerConfig(&$config, $model_id): void {
    $config['dimensions'] = [
      'type' => 'number',
      'constraints' => [
        'options' => [
          '1024',
          '512',
          '256',
        ],
      ],
      'label' => t('Dimensions'),
      'default' => 1024,
    ];
    // If its image, then its another format.
    if (strpos($model_id, 'amazon.titan-embed-image') === 0) {
      unset($config['dimensions']);
      $config['outputEmbeddingLength'] = [
        'type' => 'number',
        'constraints' => [
          'options' => [
            '1024',
            '384',
            '256',
          ],
        ],
        'label' => t('Output Embedding Length'),
        'default' => 1024,
      ];
    }
    // And if its v1, nothing.
    if (strpos($model_id, 'amazon.titan-embed-text-v1') === 0) {
      unset($config['dimensions']);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function formatInput($input = "", $image = NULL, $config = [], $model_id = ''): array {
    if ($input) {
      $payload['inputText'] = $input;
    }
    elseif ($image && strpos($model_id, 'amazon.titan-embed-image') === 0) {
      $payload['inputImage'] = $image->getAsBase64EncodedString('');
    }
    else {
      throw new AiBadRequestException('You need to give an input text or an image or you used as image in a none-image embeddings model.');
    }
    if (strpos($model_id, 'amazon.titan-embed-image') === 0) {
      $payload['embeddingConfig']['outputEmbeddingLength'] = (int) $config['outputEmbeddingLength'];
    }
    elseif (strpos($model_id, 'amazon.titan-embed-text-v1') !== 0) {
      $payload['dimensions'] = (int) $config['dimensions'];
    }
    return $payload;
  }

  /**
   * {@inheritdoc}
   */
  public static function formatOutput($output, $config = []): array {
    if (empty($output['embedding'])) {
      throw new AiBrokenOutputException('No image data found in the response.');
    }
    return $output['embedding'];
  }

  /**
   * {@inheritdoc}
   */
  public static function providerCapabilities($capabilities, $model_id): bool {
    return TRUE;
  }

}
