<?php

namespace Drupal\ai_provider_aws_bedrock\Models\TextToImage;

use Drupal\ai_provider_aws_bedrock\Attribute\BedrockModel;
use Drupal\ai\Exception\AiBrokenOutputException;
use Drupal\ai\OperationType\GenericType\ImageFile;
use Drupal\ai_provider_aws_bedrock\Models\BedrockModelInterface;
use Drupal\ai_provider_aws_bedrock\Models\ModelWithInputOutputInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * The Titan Image model.
 */
#[BedrockModel(
  id: 'amazon.titan-image-generator-v1',
  label: new TranslatableMarkup('Titan image'),
)]
class TitanImage implements BedrockModelInterface, ModelWithInputOutputInterface {

  /**
   * {@inheritdoc}
   */
  public static function providerConfig(&$config, $model_id): void {
    $config['resolution'] = [
      'type' => 'string',
      'constraints' => [
        'options' => [
          '1024x1024',
          '768x768',
          '512x512',
          '768x1152',
          '384x576',
          '1152x768',
          '576x384',
          '768x1280',
          '384x640',
          '1280x768',
          '640x384',
          '896x1152',
          '448x576',
          '1152x896',
          '576x448',
          '768x1408',
          '384x704',
          '1408x768',
          '704x384',
          '640x1408',
          '320x704',
          '1408x640',
          '704x320',
          '1152x640',
          '1173x640',
        ],
      ],
      'label' => t('Resolution'),
    ];

    $config['numberOfImages'] = [
      'type' => 'number',
      'default' => 1,
      'label' => t('Number of Images'),
      'constraints' => [
        'min' => 1,
        'max' => 5,
      ],
    ];

    $config['cfgScale'] = [
      'type' => 'float',
      'default' => 8.0,
      'label' => t('CFG Scale'),
      'constraints' => [
        'min' => 1.1,
        'max' => 10,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function formatInput($input = "", $image = NULL, $config = [], $model_id = ''): array {
    $payload = [
      'taskType' => 'TEXT_IMAGE',
      'textToImageParams' => [
        'text' => $input,
      ],
    ];
    foreach ($config as $key => $value) {
      if ($key == 'resolution' && $value) {
        $parts = explode('x', $value);
        $payload['imageGenerationConfig']['width'] = (int) $parts[0];
        $payload['imageGenerationConfig']['height'] = (int) $parts[1];
      }
      elseif ($value) {
        $payload['imageGenerationConfig'][$key] = (int) $value;
      }
    }
    return $payload;
  }

  /**
   * {@inheritdoc}
   */
  public static function formatOutput($output, $config = []): array {
    $images = [];
    if (empty($output['images'][0])) {
      throw new AiBrokenOutputException('No image data found in the response.');
    }
    foreach ($output['images'] as $data) {
      $images[] = new ImageFile(base64_decode($data), 'image/png', 'dalle.png');
    }
    return $images;
  }

  /**
   * {@inheritdoc}
   */
  public static function providerCapabilities($capabilities, $model_id): bool {
    return TRUE;
  }

}
