<?php

namespace Drupal\ai_provider_baidu\Plugin\AiProvider;

use Drupal\ai\Attribute\AiProvider;
use Drupal\ai\Base\AiProviderClientBase;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatInterface;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\Chat\ChatOutput;
use Drupal\ai_provider_baidu\BaiduClient;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the Baidu LLM.
 */
#[AiProvider(
  id: 'baidu',
  label: new TranslatableMarkup('Baidu')
)]
class BaiduProvider extends AiProviderClientBase implements ChatInterface
{

  /**
   * Baidu client
   *
   * @var BaiduClient|null
   */
  protected BaiduClient|null $client = null;

  /**
   * API Key.
   *
   * @var string
   */
  protected string $apiKey = '';

  /**
   * Run moderation call, before a normal call.
   *
   * @var bool
   */
  protected bool $moderation = TRUE;

  /**
   * {@inheritdoc}
   * @param string|null $operation_type
   * @param array $capabilities
   */
  public function isUsable(?string $operation_type = NULL, array $capabilities = []): bool
  {
    if (!$this->getConfig()->get('api_key')) {
      return FALSE;
    }

    if ($operation_type) {
      return in_array($operation_type, $this->getSupportedOperationTypes());
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedOperationTypes(): array
  {
    return ['chat'];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig(): ImmutableConfig
  {
    return $this->configFactory->get('ai_provider_baidu.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function getModelSettings(string $model_id, array $generalConfig = []): array
  {
    return $generalConfig;
  }

  /**
   * {@inheritdoc}
   */
  public function setAuthentication(mixed $authentication): void
  {
    $this->apiKey = $authentication;
  }

  /**
   * Gets the raw client.
   *
   * @param string $api_key
   *   If the API key should be hot swapped.
   *
   * @return BaiduClient
   *   The Baidu client.
   */
  public function getClient(string $api_key = '')
  {
    if ($api_key) {
      $this->setAuthentication($api_key);
    }

    $this->loadClient();
    return $this->client;
  }

  /**
   * Loads the Baidu Client with authentication if not initialized.
   */
  protected function loadClient(): void
  {
    if (!$this->client) {
      if (!$this->apiKey) {
        $this->setAuthentication($this->loadApiKey());
      }
      $this->client = \Drupal::service('ai_provider_baidu.client');
      $this->client->setApiToken($this->apiKey);
    }
  }

  /**
   * {@inheritdoc}
   * @param string|null $operation_type
   * @param array $capabilities
   */
  public function getConfiguredModels(?string $operation_type = NULL, array $capabilities = []): array
  {
    switch ($operation_type) {
      case 'chat':
        $models = [
          // ERNIE Series - Flagship Model
          'ernie-4.5-turbo-128k',
          'ernie-4.5-turbo-128k-preview',
          'ernie-4.5-turbo-32k',
          'ernie-4.5-turbo-vl-preview',
          'ernie-4.5-turbo-vl',
          'ernie-4.5-turbo-vl-32k',
          'ernie-4.5-turbo-vl-32k-preview',
          'ernie-4.5-8k-preview',
          // ERNIE Series - Main model
          'ernie-speed-128k',
          'ernie-speed-8k',
          'ernie-speed-pro-128k',
          'ernie-lite-8k',
          'ernie-lite-pro-128k',
          // ERNIE Series - Lightweight Models
          'ernie-tiny-8k',
          // DeepSeek Series
          'DeepSeek-V3.1-250821',
          // Others
          'Kimi-K2-Instruct',
        ];
        break;
      default:
        $models = [];
    }
    return array_combine($models, $models);
  }

  /**
   * Load API key from key module.
   *
   * @return string
   *   The API key.
   */
  protected function loadApiKey(): string
  {
    return $this->keyRepository->getKey($this->getConfig()->get('api_key'))
      ->getKeyValue();
  }

  public function chat(array|string|ChatInput $input, string $model_id, array $tags = []): ChatOutput
  {
    $this->loadClient();

    $response = $this->client->textGeneration('/v2/chat/completions', $model_id, $input);
    $data = Json::decode($response);
    $message = new ChatMessage($data['choices'][0]['message']['role'] ?? '', $data['choices'][0]['message']['content'] ?? '');

    return new ChatOutput($message, $response, []);
  }

}
