/**
 * @file
 * Browser-based AI provider using Chrome's LanguageModel API.
 */

const BrowserAiProvider = (function () {
  'use strict';

  /**
   * Gets settings from Drupal or falls back to defaults.
   *
   * @returns {Object} The configuration settings.
   */
  function getDrupalSettings() {
    const settings = window.drupalSettings?.aiProviderBrowser || {};

    return {
      timeout: (settings.timeout || 60) * 1000,
      pollingInterval: (settings.pollingInterval || 5) * 1000,
    };
  }

  const drupalSettings = getDrupalSettings();

  const CONFIG = {
    endpoints: {
      list: '/ai-provider-browser/chat-requests',
      consume: '/ai-provider-browser/chat-requests/consume',
      store: '/ai-provider-browser/chat-requests/store',
    },
    pollingInterval: drupalSettings.pollingInterval,
    timeout: drupalSettings.timeout,
    requestStatus: {
      received: 'received',
      processing: 'processing',
      completed: 'completed',
      failed: 'failed',
    },
  };

  let pollingTimer = null;
  let isProcessing = false;

  /**
   * Checks if the LanguageModel API is available and ready.
   *
   * @returns {Promise<string>} The availability status.
   */
  async function checkAvailability() {
    if (typeof LanguageModel === 'undefined') {
      return 'unavailable';
    }

    try {
      return await LanguageModel.availability();
    } catch (error) {
      console.error('[BrowserAiProvider] Failed to check availability:', error);
      return 'unavailable';
    }
  }

  let downloadPromptElement = null;

  /**
   * Creates the download prompt UI element.
   *
   * @returns {HTMLElement} The prompt container element.
   */
  function createDownloadPromptElement() {
    const container = document.createElement('div');
    container.className = 'browser-ai-download-prompt';
    container.innerHTML = `
      <div class="browser-ai-download-prompt__content">
        <h3 class="browser-ai-download-prompt__title">AI Model Required</h3>
        <p class="browser-ai-download-prompt__message">
          The browser AI provider requires downloading a language model to function.
          This is a one-time download.
        </p>
        <div class="browser-ai-download-prompt__progress" style="display: none;">
          <div class="browser-ai-download-prompt__progress-bar">
            <div class="browser-ai-download-prompt__progress-fill"></div>
          </div>
          <span class="browser-ai-download-prompt__progress-text">0%</span>
        </div>
        <div class="browser-ai-download-prompt__actions">
          <button type="button" class="browser-ai-download-prompt__button browser-ai-download-prompt__button--primary">
            Download Model
          </button>
          <button type="button" class="browser-ai-download-prompt__button browser-ai-download-prompt__button--secondary">
            Dismiss
          </button>
        </div>
      </div>
    `;

    // Apply inline styles for positioning (can be overridden by CSS).
    Object.assign(container.style, {
      position: 'fixed',
      bottom: '20px',
      right: '20px',
      zIndex: '10000',
      backgroundColor: '#fff',
      borderRadius: '8px',
      boxShadow: '0 4px 12px rgba(0, 0, 0, 0.15)',
      padding: '16px',
      maxWidth: '320px',
      fontFamily: 'system-ui, -apple-system, sans-serif',
    });

    return container;
  }

  /**
   * Shows the download prompt to the user.
   */
  function showDownloadPrompt() {
    if (downloadPromptElement) {
      return;
    }

    downloadPromptElement = createDownloadPromptElement();
    document.body.appendChild(downloadPromptElement);

    const downloadButton = downloadPromptElement.querySelector(
      '.browser-ai-download-prompt__button--primary'
    );
    const dismissButton = downloadPromptElement.querySelector(
      '.browser-ai-download-prompt__button--secondary'
    );

    downloadButton.addEventListener('click', downloadModel);
    dismissButton.addEventListener('click', hideDownloadPrompt);
  }

  /**
   * Hides and removes the download prompt.
   */
  function hideDownloadPrompt() {
    if (downloadPromptElement) {
      downloadPromptElement.remove();
      downloadPromptElement = null;
    }
  }

  /**
   * Updates the download progress UI.
   *
   * @param {number} progress - Progress value between 0 and 1.
   */
  function updateDownloadProgress(progress) {
    if (!downloadPromptElement) {
      return;
    }

    const progressContainer = downloadPromptElement.querySelector(
      '.browser-ai-download-prompt__progress'
    );
    const progressFill = downloadPromptElement.querySelector(
      '.browser-ai-download-prompt__progress-fill'
    );
    const progressText = downloadPromptElement.querySelector(
      '.browser-ai-download-prompt__progress-text'
    );
    const actions = downloadPromptElement.querySelector(
      '.browser-ai-download-prompt__actions'
    );

    progressContainer.style.display = 'block';
    actions.style.display = 'none';

    const percentage = Math.round(progress * 100);
    progressFill.style.width = `${percentage}%`;
    progressFill.style.backgroundColor = '#4285f4';
    progressFill.style.height = '100%';
    progressFill.style.borderRadius = '4px';
    progressFill.style.transition = 'width 0.3s ease';

    progressText.textContent = `${percentage}%`;

    // Style the progress bar container.
    const progressBar = downloadPromptElement.querySelector(
      '.browser-ai-download-prompt__progress-bar'
    );
    Object.assign(progressBar.style, {
      backgroundColor: '#e0e0e0',
      borderRadius: '4px',
      height: '8px',
      overflow: 'hidden',
      marginBottom: '8px',
    });
  }

  /**
   * Shows download complete state and starts polling.
   */
  function onDownloadComplete() {
    if (downloadPromptElement) {
      const message = downloadPromptElement.querySelector(
        '.browser-ai-download-prompt__message'
      );
      const progressContainer = downloadPromptElement.querySelector(
        '.browser-ai-download-prompt__progress'
      );

      message.textContent = 'Download complete! The AI provider is now ready.';
      progressContainer.style.display = 'none';

      setTimeout(() => {
        hideDownloadPrompt();
        startPolling();
      }, 2000);
    } else {
      startPolling();
    }
  }

  /**
   * Initiates download of the language model.
   */
  async function downloadModel() {
    if (!navigator.userActivation?.isActive) {
      console.warn('[BrowserAiProvider] User activation required to download model');
      return;
    }

    console.log('[BrowserAiProvider] Starting model download...');
    updateDownloadProgress(0);

    try {
      await LanguageModel.create({
        monitor(m) {
          m.addEventListener('downloadprogress', (e) => {
            console.log(`[BrowserAiProvider] Download progress: ${(e.loaded * 100).toFixed(1)}%`);
            updateDownloadProgress(e.loaded);
          });
        },
      });

      console.log('[BrowserAiProvider] Download complete');
      onDownloadComplete();
    } catch (error) {
      console.error('[BrowserAiProvider] Failed to download model:', error);

      if (downloadPromptElement) {
        const message = downloadPromptElement.querySelector(
          '.browser-ai-download-prompt__message'
        );
        const progressContainer = downloadPromptElement.querySelector(
          '.browser-ai-download-prompt__progress'
        );
        const actions = downloadPromptElement.querySelector(
          '.browser-ai-download-prompt__actions'
        );

        message.textContent = 'Download failed. Please try again.';
        message.style.color = '#d93025';
        progressContainer.style.display = 'none';
        actions.style.display = 'block';
      }
    }
  }

  /**
   * Converts a base64 data URI to a Blob.
   *
   * @param {string} dataUri - The base64 data URI (e.g., "data:image/png;base64,...").
   * @returns {Blob} The converted Blob object.
   */
  function base64ToBlob(dataUri) {
    const [header, base64Data] = dataUri.split(',');
    const mimeMatch = header.match(/data:([^;]+)/);
    const mimeType = mimeMatch ? mimeMatch[1] : 'application/octet-stream';

    const binaryString = atob(base64Data);
    const bytes = new Uint8Array(binaryString.length);

    for (let i = 0; i < binaryString.length; i++) {
      bytes[i] = binaryString.charCodeAt(i);
    }

    return new Blob([bytes], { type: mimeType });
  }

  /**
   * Creates a language model session with the given initial prompts.
   *
   * @param {Array} initialPrompts - Initial prompts for the session.
   * @param {boolean} hasImages - Whether the session will include images.
   * @returns {Promise<Object>} The language model session.
   */
  async function createSession(initialPrompts = [], hasImages = false) {
    const defaultPrompts = [
      { role: 'system', content: 'You are a helpful and friendly assistant.' },
    ];

    const options = {
      initialPrompts: initialPrompts.length > 0 ? initialPrompts : defaultPrompts,
    };

    // Declare expected inputs when images are used.
    if (hasImages) {
      options.expectedInputs = [
        { type: 'text' },
        { type: 'image' },
      ];
    }

    return await LanguageModel.create(options);
  }

  /**
   * Wraps a promise with a timeout.
   *
   * @param {Promise} promise - The promise to wrap.
   * @param {number} ms - Timeout in milliseconds.
   * @returns {Promise} The wrapped promise.
   */
  function withTimeout(promise, ms) {
    return Promise.race([
      promise,
      new Promise((_, reject) =>
        setTimeout(() => reject(new Error(`Timeout after ${ms}ms`)), ms)
      ),
    ]);
  }

  /**
   * Sends a prompt to the language model.
   *
   * @param {string|Array} prompt - The user prompt (string or multimodal content array).
   * @param {Array} initialPrompts - Initial prompts including system messages.
   * @param {Object|null} responseConstraint - JSON Schema for structured output.
   * @param {boolean} hasImages - Whether the prompt contains images.
   * @returns {Promise<string>} The model's response.
   */
  async function sendPrompt(prompt, initialPrompts = [], responseConstraint = null, hasImages = false) {
    const session = await createSession(initialPrompts, hasImages);
    const options = {};

    // Add responseConstraint for structured output if schema is provided.
    if (responseConstraint && Object.keys(responseConstraint).length > 0) {
      options.responseConstraint = responseConstraint;
    }

    // For multimodal content (array), use the message format with role and content.
    if (Array.isArray(prompt)) {
      const messagePayload = [
        {
          role: 'user',
          content: prompt,
        },
      ];
      return await withTimeout(session.prompt(messagePayload, options), CONFIG.timeout);
    }

    // For plain text prompts, use the simple string format.
    return await withTimeout(session.prompt(prompt, options), CONFIG.timeout);
  }

  /**
   * Fetches pending chat requests from the server.
   *
   * @returns {Promise<Object|null>} The request data or null on error.
   */
  async function fetchPendingRequests() {
    try {
      const response = await fetch(CONFIG.endpoints.list);

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}`);
      }

      return await response.json();
    } catch (error) {
      console.error('[BrowserAiProvider] Failed to fetch requests:', error);
      return null;
    }
  }

  /**
   * Marks a request as being processed.
   *
   * @param {string} uuid - The request UUID.
   * @returns {Promise<boolean>} True if successful.
   */
  async function consumeRequest(uuid) {
    try {
      const response = await fetch(CONFIG.endpoints.consume, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ uuid }),
      });

      const result = await response.json();
      return result.success === true;
    } catch (error) {
      console.error('[BrowserAiProvider] Failed to consume request:', error);
      return false;
    }
  }

  /**
   * Stores the AI response for a request.
   *
   * @param {string} uuid - The request UUID.
   * @param {string} responseMessage - The AI response.
   * @returns {Promise<boolean>} True if successful.
   */
  async function storeResponse(uuid, responseMessage) {
    try {
      const response = await fetch(CONFIG.endpoints.store, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          uuid,
          response_message: responseMessage,
        }),
      });

      const result = await response.json();
      return result.success === true;
    } catch (error) {
      console.error('[BrowserAiProvider] Failed to store response:', error);
      return false;
    }
  }

  /**
   * Builds a multimodal content array from text and images.
   *
   * @param {string} text - The text content.
   * @param {Array} images - Array of image objects with base64 data.
   * @returns {Array} Content array for the Prompt API.
   */
  function buildMultimodalContent(text, images) {
    const content = [];

    // Add text content first.
    if (text) {
      content.push({ type: 'text', value: text });
    }

    // Add image content.
    if (images && images.length > 0) {
      for (const image of images) {
        if (image.base64) {
          const blob = base64ToBlob(image.base64);
          content.push({ type: 'image', value: blob });
        }
      }
    }

    return content;
  }

  /**
   * Extracts messages from a chat request, separated by role.
   *
   * @param {Object} request - The chat request object.
   * @returns {Object} Object containing initialPrompts, userPrompt, responseConstraint, and hasImages.
   */
  function extractMessagesFromRequest(request) {
    const messages = request?.chatInput?.messages;

    const initialPrompts = [];
    let userPrompt = null;
    let hasImages = false;

    // Extract JSON schema for structured output (responseConstraint).
    const jsonSchema = request?.chatInput?.chat_structured_json_schema || null;
    // The schema may be wrapped with metadata (name, strict, schema) or be the schema directly.
    const responseConstraint = jsonSchema?.schema || jsonSchema;

    // Add system prompt if available.
    if (request?.chatInput?.system_prompt) {
      initialPrompts.push({
        role: 'system',
        content: request.chatInput.system_prompt,
      });
    }

    for (const message of messages) {
      const role = message.role;
      const text = message.text || '';
      const images = message.images || [];

      // Check if this message has images.
      if (images.length > 0) {
        hasImages = true;
      }

      if (role === 'user') {
        // Build multimodal content if images are present, otherwise use plain text.
        if (images.length > 0) {
          userPrompt = buildMultimodalContent(text, images);
        } else {
          userPrompt = text;
        }
      } else if (role === 'assistant') {
        // Include assistant messages in the conversation history.
        initialPrompts.push({ role: 'assistant', content: text });
      }
    }

    return { initialPrompts, userPrompt, responseConstraint, hasImages };
  }

  /**
   * Processes a single chat request.
   *
   * @param {Object} request - The chat request to process.
   */
  async function processRequest(request) {
    const uuid = request.requestUuid;
    const { initialPrompts, userPrompt, responseConstraint, hasImages } = extractMessagesFromRequest(request);

    if (!userPrompt) {
      console.error('[BrowserAiProvider] No user prompt found in request:', uuid);
      return;
    }

    const consumed = await consumeRequest(uuid);

    if (!consumed) {
      console.error('[BrowserAiProvider] Failed to consume request:', uuid);
      return;
    }

    try {
      const response = await sendPrompt(userPrompt, initialPrompts, responseConstraint, hasImages);
      await storeResponse(uuid, response);
    } catch (error) {
      console.error('[BrowserAiProvider] Failed to process prompt:', error);
    }
  }

  /**
   * Polls for and processes pending requests.
   */
  async function poll() {
    if (isProcessing) {
      return;
    }

    const data = await fetchPendingRequests();

    if (!data?.data?.length) {
      return;
    }

    const pendingRequest = data.data.find(
      (req) => req.status === CONFIG.requestStatus.received
    );

    if (!pendingRequest) {
      return;
    }

    isProcessing = true;

    try {
      await processRequest(pendingRequest);
    } finally {
      isProcessing = false;
    }
  }

  /**
   * Starts the polling loop.
   */
  function startPolling() {
    if (pollingTimer) {
      return;
    }

    pollingTimer = setInterval(poll, CONFIG.pollingInterval);
    poll();
  }

  /**
   * Stops the polling loop.
   */
  function stopPolling() {
    if (pollingTimer) {
      clearInterval(pollingTimer);
      pollingTimer = null;
    }
  }

  /**
   * Initializes the browser AI provider.
   */
  async function init() {
    const availability = await checkAvailability();
    console.log(availability);
    switch (availability) {
      case 'available':
      case 'readily':
        startPolling();
        break;

      case 'unavailable':
      case 'downloadable':
        showDownloadPrompt();
        break;

      case 'downloading':
        // Model is already being downloaded, show progress UI.
        showDownloadPrompt();
        // Update message to indicate download is in progress.
        if (downloadPromptElement) {
          const message = downloadPromptElement.querySelector(
            '.browser-ai-download-prompt__message'
          );
          const actions = downloadPromptElement.querySelector(
            '.browser-ai-download-prompt__actions'
          );
          message.textContent = 'The AI model is currently downloading. Please wait...';
          actions.style.display = 'none';
        }
        // Poll for availability to detect when download completes.
        waitForModelReady();
        break;

      case 'unavailable':
      default:
        console.warn('[BrowserAiProvider] LanguageModel API not available');
        break;
    }
  }

  /**
   * Polls for model availability during an in-progress download.
   */
  async function waitForModelReady() {
    const checkInterval = setInterval(async () => {
      const status = await checkAvailability();

      if (status === 'available' || status === 'readily') {
        clearInterval(checkInterval);
        onDownloadComplete();
      }
    }, 2000);
  }

  // Auto-initialize when DOM is ready.
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

  return {
    init,
    startPolling,
    stopPolling,
    checkAvailability,
    showDownloadPrompt,
    hideDownloadPrompt,
  };
})();
