<?php

namespace Drupal\ai_provider_browser;

use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\Chat\ChatOutput;

/**
 * Simple implementation of ChatMessageRequestInterface.
 */
class ChatMessageRequest implements ChatMessageRequestInterface {

  /**
   * Constructs a ChatMessageRequest object.
   *
   * @param string $requestUuid
   *   The request uuid.
   * @param \Drupal\ai\OperationType\Chat\ChatInput|null $chatInput
   *   The chat input messages.
   * @param \Drupal\ai\OperationType\Chat\ChatOutput|null $chatOutput
   *   The chat output message.
   * @param string $status
   *   The current status.
   * @param string $errorMessage
   *   The error message if any.
   * @param int $createdTime
   *   The created time.
   * @param int $lastUpdatedTime
   *   The last updated time.
   * @param int $runningStartTime
   *   The running start time.
   */
  public function __construct(
    protected string $requestUuid = '',
    protected ?ChatInput $chatInput = NULL,
    protected ?ChatOutput $chatOutput = NULL,
    protected string $status = self::STATUS_RECEIVED,
    protected string $errorMessage = '',
    protected int $createdTime = 0,
    protected int $lastUpdatedTime = 0,
    protected int $runningStartTime = 0,
  ) {
    // Set both created and last updated time to now, if empty.
    if (empty($this->createdTime)) {
      $this->createdTime = time();
    }
    if (empty($this->lastUpdatedTime)) {
      $this->lastUpdatedTime = $this->createdTime;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getRequestUuid(): string {
    return $this->requestUuid;
  }

  /**
   * {@inheritdoc}
   */
  public function setRequestUuid(string $request_uuid): static {
    // A request uuid should be immutable once set.
    if (!empty($this->requestUuid)) {
      throw new \LogicException('The request UUID is already set and cannot be changed.');
    }
    $this->requestUuid = $request_uuid;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getChatInput(): ChatInput {
    return $this->chatInput ?? new ChatInput([]);
  }

  /**
   * {@inheritdoc}
   */
  public function setChatInput(ChatInput $input): static {
    // ChatInput can only be set once the request uuid is set.
    if (empty($this->requestUuid)) {
      throw new \LogicException('The request UUID must be set before setting the ChatInput.');
    }
    $this->chatInput = $input;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getChatOutput(): ChatOutput {
    return $this->chatOutput ?? new ChatOutput(new ChatMessage('assistant', 'hello'), '', []);
  }

  /**
   * {@inheritdoc}
   */
  public function setChatOutput(ChatOutput $output): static {
    // ChatOutput can only be set once the request uuid is set.
    if (empty($this->requestUuid)) {
      throw new \LogicException('The request UUID must be set before setting the ChatOutput.');
    }
    $this->chatOutput = $output;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getStatus(): string {
    return $this->status;
  }

  /**
   * {@inheritdoc}
   */
  public function setStatus(string $status): static {
    // If status is set to processing, set the timestamp.
    $this->status = $status;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getErrorMessage(): string {
    return $this->errorMessage;
  }

  /**
   * {@inheritdoc}
   */
  public function setErrorMessage(string $message): static {
    $this->errorMessage = $message;
    return $this;
  }

  /**
   * {@inheritDoc}
   */
  public function getCreatedTime(): int {
    return $this->createdTime;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime(int $created): static {
    $this->createdTime = $created;
    return $this;
  }

  /**
   * {@inheritDoc}
   */
  public function getUpdatedTime(): int {
    return $this->lastUpdatedTime;
  }

  /**
   * {@inheritDoc}
   */
  public function setUpdatedTime(int $updated): static {
    $this->lastUpdatedTime = $updated;
    return $this;
  }

  /**
   * {@inheritDoc}
   */
  public function getRunningStartTime(): int {
    return $this->runningStartTime;
  }

  /**
   * {@inheritDoc}
   */
  public function setRunningStartTime(int $started): static {
    $this->runningStartTime = $started;
    return $this;
  }

  /**
   * {@inheritDoc}
   */
  public function getRunningTime(): int {
    return time() - $this->runningStartTime;
  }

  /**
   * {@inheritdoc}
   */
  public static function fromArray(array $data): static {
    $chat_input = isset($data['chatInput']) ? ChatInput::fromArray($data['chatInput']) : NULL;
    $chat_output = isset($data['chatOutput']) ? ChatOutput::fromArray($data['chatOutput']) : NULL;

    return new static(
      requestUuid: $data['requestUuid'] ?? '',
      chatInput: $chat_input,
      chatOutput: $chat_output,
      status: $data['status'] ?? self::STATUS_RECEIVED,
      errorMessage: $data['errorMessage'] ?? '',
      createdTime: $data['createdTime'] ?? 0,
      lastUpdatedTime: $data['lastUpdatedTime'] ?? 0,
      runningStartTime: $data['runningStartTime'] ?? 0,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function toArray(): array {
    $array = [
      'requestUuid' => $this->requestUuid,
      'status' => $this->status,
      'errorMessage' => $this->errorMessage,
      'createdTime' => $this->createdTime,
      'lastUpdatedTime' => $this->lastUpdatedTime,
      'runningStartTime' => $this->runningStartTime,
    ];

    if ($this->chatInput !== NULL) {
      $array['chatInput'] = $this->chatInput->toArray();
    }

    if ($this->chatOutput !== NULL) {
      $array['chatOutput'] = $this->chatOutput->toArray();
    }

    return $array;
  }

}
