<?php

namespace Drupal\ai_provider_browser;

use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\ai\OperationType\Chat\ChatInput;

/**
 * Service for managing chat message requests in private temp storage.
 */
class ChatMessageStorage {

  /**
   * The collection name for storing chat message requests.
   */
  const COLLECTION = 'ai_provider_browser_chat_requests';

  /**
   * The private temp store.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStore
   */
  protected $tempStore;

  /**
   * Constructs a ChatMessageStorage object.
   *
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The private temp store factory.
   * @param \Drupal\Component\Uuid\UuidInterface $uuid
   *   The UUID service.
   */
  public function __construct(
    PrivateTempStoreFactory $temp_store_factory,
    protected UuidInterface $uuid,
  ) {
    $this->tempStore = $temp_store_factory->get(self::COLLECTION);
  }

  /**
   * Adds a new request to the list for the current user.
   *
   * Creates a UUID for the request and stores it.
   *
   * @param \Drupal\ai\OperationType\Chat\ChatInput $chat_input
   *   The chat input to store.
   *
   * @return \Drupal\ai_provider_browser\ChatMessageRequestInterface
   *   The created chat message request with UUID.
   *
   * @throws \Drupal\Core\TempStore\TempStoreException
   */
  public function addRequest(ChatInput $chat_input): ChatMessageRequestInterface {
    $request_uuid = $this->uuid->generate();

    $request = new ChatMessageRequest(
      requestUuid: $request_uuid,
      chatInput: $chat_input,
      status: ChatMessageRequestInterface::STATUS_RECEIVED,
    );

    // Get existing requests and add the new one.
    $requests = $this->getAllRequests();
    $requests[$request_uuid] = $request->toArray();

    $this->tempStore->set('requests', $requests);

    return $request;
  }

  /**
   * Finds and updates a request based on UUID.
   *
   * @param string $uuid
   *   The UUID of the request to update.
   * @param \Drupal\ai_provider_browser\ChatMessageRequestInterface $request
   *   The updated request object.
   *
   * @return bool
   *   TRUE if the request was found and updated, FALSE otherwise.
   *
   * @throws \Drupal\Core\TempStore\TempStoreException
   */
  public function updateRequest(string $uuid, ChatMessageRequestInterface $request): bool {
    $requests = $this->getAllRequests();

    if (!isset($requests[$uuid])) {
      return FALSE;
    }

    // Update the last updated time.
    $request->setUpdatedTime(time());
    $requests[$uuid] = $request->toArray();

    $this->tempStore->set('requests', $requests);

    return TRUE;
  }

  /**
   * Finds a request by UUID.
   *
   * @param string $uuid
   *   The UUID of the request to find.
   *
   * @return \Drupal\ai_provider_browser\ChatMessageRequestInterface|null
   *   The chat message request or NULL if not found.
   */
  public function getRequest(string $uuid): ?ChatMessageRequestInterface {
    $requests = $this->getAllRequests();

    if (!isset($requests[$uuid])) {
      return NULL;
    }

    return ChatMessageRequest::fromArray($requests[$uuid]);
  }

  /**
   * Removes a request from the list based on UUID.
   *
   * @param string $uuid
   *   The UUID of the request to remove.
   *
   * @return bool
   *   TRUE if the request was found and removed, FALSE otherwise.
   *
   * @throws \Drupal\Core\TempStore\TempStoreException
   */
  public function removeRequest(string $uuid): bool {
    $requests = $this->getAllRequests();

    if (!isset($requests[$uuid])) {
      return FALSE;
    }

    unset($requests[$uuid]);
    $this->tempStore->set('requests', $requests);

    return TRUE;
  }

  /**
   * Gets all requests for the current user.
   *
   * @return array
   *   Array of request data keyed by UUID.
   */
  public function getAllRequests(): array {
    $requests = $this->tempStore->get('requests');
    return is_array($requests) ? $requests : [];
  }

  /**
   * Clears all requests for the current user.
   *
   * @throws \Drupal\Core\TempStore\TempStoreException
   */
  public function clearAllRequests(): void {
    $this->tempStore->delete('requests');
  }

  /**
   * Sets timeout status on processes that has past.
   */
  public function setTimedOutRequests(): void {
    $requests = $this->getAllRequests();
    $current_time = time();
    $timeout_duration = 60; // 1 minute timeout.

    foreach ($requests as $uuid => $request_data) {
      $request = ChatMessageRequest::fromArray($request_data);
      if ($request->getStatus() === ChatMessageRequestInterface::STATUS_PROCESSING) {
        $elapsed_time = $current_time - $request->getUpdatedTime();
        if ($elapsed_time > $timeout_duration) {
          $request->setStatus(ChatMessageRequestInterface::STATUS_TIME_OUT);
          $this->updateRequest($uuid, $request);
        }
      }
    }
  }

}
