<?php

namespace Drupal\ai_provider_browser\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\Chat\ChatOutput;
use Drupal\ai_provider_browser\ChatMessageRequestInterface;
use Drupal\ai_provider_browser\ChatMessageStorage;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for handling chat message requests.
 */
class ChatMessageRequestHandler extends ControllerBase {

  /**
   * Constructs a ChatMessageRequestHandler object.
   *
   * @param \Drupal\ai_provider_browser\ChatMessageStorage $chatMessageStorage
   *   The chat message storage service.
   */
  public function __construct(
    protected ChatMessageStorage $chatMessageStorage,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('ai_provider_browser.chat_message_storage'),
    );
  }

  /**
   * Returns a list of chat message requests.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response containing the list of chat message requests.
   */
  public function listRequests(): JsonResponse {
    // Set time out.
    $this->chatMessageStorage->setTimedOutRequests();
    // Then get all requests.
    $requests = $this->chatMessageStorage->getAllRequests();

    if (empty($requests)) {
      return new JsonResponse([
        'data' => [],
        'count' => 0,
      ]);
    }

    // Get the last request only.
    $one_request = end($requests);

    return new JsonResponse([
      'data' => [$one_request],
      'count' => count($requests),
    ]);
  }

  /**
   * Consumes a UUID and sets it to processing status.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response verifying that the request is getting consumed.
   */
  public function consumeRequest(Request $request): JsonResponse {
    $data = json_decode($request->getContent(), TRUE);
    $uuid = $data['uuid'] ?? NULL;

    if (!$uuid) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'No UUID provided.',
      ]);
    }

    $chat_request = $this->chatMessageStorage->getRequest($uuid);

    if (!$chat_request) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Request not found.',
      ]);
    }

    $chat_request->setStatus(ChatMessageRequestInterface::STATUS_PROCESSING);
    $this->chatMessageStorage->updateRequest($uuid, $chat_request);

    return new JsonResponse([
      'success' => TRUE,
    ]);
  }

  /**
   * Stores a response message.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response from the browser provider.
   */
  public function storeResponse(Request $request): JsonResponse {
    $data = json_decode($request->getContent(), TRUE);
    $uuid = $data['uuid'] ?? NULL;

    if (!$uuid) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'No UUID provided.',
      ]);
    }

    $chat_request = $this->chatMessageStorage->getRequest($uuid);

    if (!$chat_request) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Request not found.',
      ]);
    }

    $response_text = $data['response_message'] ?? '';
    $chat_output = new ChatOutput(
      new ChatMessage('assistant', $response_text),
      [],
      []
    );

    $chat_request->setChatOutput($chat_output);
    $chat_request->setStatus(ChatMessageRequestInterface::STATUS_COMPLETED);
    $this->chatMessageStorage->updateRequest($uuid, $chat_request);

    return new JsonResponse([
      'success' => TRUE,
    ]);
  }

}
