<?php

namespace Drupal\ai_provider_browser\Plugin\AiProvider;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\AiProvider;
use Drupal\ai\Base\AiProviderClientBase;
use Drupal\ai\Exception\AiRequestErrorException;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatInterface;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\Chat\ChatOutput;
use Drupal\ai\Traits\OperationType\ChatTrait;
use Drupal\ai_provider_browser\ChatMessageRequestInterface;

/**
 * Plugin implementation of the 'browser' provider.
 */
#[AiProvider(
  id: 'browser',
  label: new TranslatableMarkup('Browser Client'),
)]
class BrowserProvider extends AiProviderClientBase implements
  ContainerFactoryPluginInterface,
  ChatInterface {

  use ChatTrait;

  /**
   * {@inheritdoc}
   */
  public function getConfiguredModels(?string $operation_type = NULL, array $capabilities = []): array {
    return [
      'chrome' => 'Google Chrome',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isUsable(?string $operation_type = NULL, array $capabilities = []): bool {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedOperationTypes(): array {
    return [
      'chat',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getModelSettings(string $model_id, array $generalConfig = []): array {
    return $generalConfig;
  }

  /**
   * {@inheritdoc}
   */
  public function setAuthentication(mixed $authentication): void {
  }

  /**
   * {@inheritdoc}
   */
  public function chat(array|string|ChatInput $input, string $model_id, array $tags = []): ChatOutput {
    $storage = \Drupal::service('ai_provider_browser.chat_message_storage');
    $request = $storage->addRequest($input);
    // Get the uuid.
    $uuid = $request->getRequestUuid();
    // Start polling against that UUID for it being completed. Timeout 60s.
    while (TRUE) {
      sleep(1);
      $updated_request = $storage->getRequest($uuid);
      if ($updated_request && $updated_request->getStatus() === ChatMessageRequestInterface::STATUS_COMPLETED) {
        $chat_output = $updated_request->getChatOutput();
        // We can delete the request here if desired.
        $storage->removeRequest($uuid);
        if ($chat_output) {
          return $chat_output;
        }
        else {
          break;
        }
      }
      if ($updated_request && in_array($updated_request->getStatus(), [
        ChatMessageRequestInterface::STATUS_FAILED,
        ChatMessageRequestInterface::STATUS_TIME_OUT,
      ])) {
        throw new AiRequestErrorException('The request has failed or timed out.');
      }
    }

    $message = new ChatMessage('assistant', $uuid);
    return new ChatOutput($message, [], []);
  }

}
