<?php

namespace Drupal\ai_provider_bytedance\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\ai\AiProviderPluginManager;
use Drupal\key\KeyRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure ByteDance API access.
 */
class ByteDanceConfigForm extends ConfigFormBase {

  /**
   * Config settings.
   */
  const CONFIG_NAME = 'ai_provider_bytedance.settings';

  /**
   * Default provider ID.
   */
  const string PROVIDER_ID = 'bytedance';

  /**
   * Constructs a new ByteDanceConfigForm object.
   */
  final public function __construct(
    private readonly AiProviderPluginManager $aiProviderManager,
    private readonly KeyRepositoryInterface $keyRepository,
  ) {}

  /**
   * {@inheritdoc}
   *
   * @return static
   */
  final public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('ai.provider'),
      $container->get('key.repository'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'bytedance_settings';
  }

  /**
   * {@inheritdoc}
   *
   * @return array<string>
   *   The editable config names.
   */
  protected function getEditableConfigNames(): array {
    return [
      static::CONFIG_NAME,
    ];
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array<string, mixed>
   *   The form render array.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config(static::CONFIG_NAME);

    $form['api_key'] = [
      '#type'          => 'key_select',
      '#title'         => $this->t('ByteDance API Key'),
      '#description'   => $this->t('A valid API key is required to use ByteDance ModelArk services.'),
      '#default_value' => $config->get('api_key'),
      '#required'      => TRUE,
    ];

    $form['host'] = [
      '#type'          => 'url',
      '#title'         => $this->t('Base URL'),
      '#description'   => $this->t('The base URL for the API. Default: https://ark.ap-southeast.bytepluses.com/api/v3'),
      '#default_value' => $config->get('host'),
      '#required'      => FALSE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // Validate the api key against model listing.
    $key = $form_state->getValue('api_key');
    if (empty($key)) {
      $form_state->setErrorByName('api_key', $this->t('The API key is required. Please select a valid key from the list.'));

      return;
    }
    $api_key = $this->keyRepository->getKey($key)?->getKeyValue();
    if (!$api_key) {
      $form_state->setErrorByName('api_key', $this->t('The API key is invalid. Please double-check that the selected key has a value.'));

      return;
    }

    $provider = $this->aiProviderManager->createInstance(static::PROVIDER_ID);

    // Temporarily set the API key and host for validation.
    // @phpstan-ignore-next-line - ProviderProxy uses __call() magic method
    $provider->setAuthentication($api_key);
    $host = $form_state->getValue('host');

    // Clean up the host URL.
    if (!empty($host)) {
      // @phpstan-ignore-next-line - ProviderProxy uses __call() magic method
      $provider->setConfiguration(['host' => $host]);
      // Update the form value with the cleaned host so it saves correctly.
      $form_state->setValue('host', $host);
    }

    try {
      // Test connectivity by attempting to get configured models.
      // @phpstan-ignore-next-line - ProviderProxy uses __call() magic method
      $provider->getConfiguredModels('text_to_image');
    }
    catch (\Exception $e) {
      $form_state->setErrorByName('api_key', $this->t('The selected API key is not working or the host is unreachable. Error: @error', ['@error' => $e->getMessage()]));
    }
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @phpstan-param array<string, mixed> $form
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Retrieve the configuration.
    $this->config(static::CONFIG_NAME)
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('host', $form_state->getValue('host'))
      ->save();

    // Set default models.
    $this->setDefaultModels();
    // @phpstan-ignore-next-line - Parent method signature doesn't specify array type
    parent::submitForm($form, $form_state);
  }

  /**
   * Set default models for the AI provider.
   */
  private function setDefaultModels(): void {
    // Create provider instance.
    $provider = $this->aiProviderManager->createInstance(static::PROVIDER_ID);

    // Check if getSetupData() method exists and is callable.
    if (is_callable([$provider, 'getSetupData'])) {
      // Fetch setup data.
      $setup_data = $provider->getSetupData();

      // Ensure the setup data is valid.
      if (!empty($setup_data) && is_array($setup_data) && !empty($setup_data['default_models']) && is_array($setup_data['default_models'])) {
        // Loop through and set default models for each operation type.
        foreach ($setup_data['default_models'] as $op_type => $model_id) {
          $this->aiProviderManager->defaultIfNone($op_type, static::PROVIDER_ID, $model_id);
        }
      }
    }
  }

}
