<?php

namespace Drupal\ai_provider_bytedance\Plugin\AiProvider;

use Drupal\ai\Attribute\AiProvider;
use Drupal\ai\Base\OpenAiBasedProviderClientBase;
use Drupal\ai\Exception\AiQuotaException;
use Drupal\ai\Exception\AiRateLimitException;
use Drupal\ai\Exception\AiRequestErrorException;
use Drupal\ai\Exception\AiResponseErrorException;
use Drupal\ai\Exception\AiSetupFailureException;
use Drupal\ai\Exception\AiUnsafePromptException;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\Chat\ChatOutput;
use Drupal\ai\OperationType\Chat\OpenAiTypeStreamedChatMessageIterator;
use Drupal\ai\OperationType\Chat\Tools\ToolsFunctionOutput;
use Drupal\ai\OperationType\GenericType\ImageFile;
use Drupal\ai\OperationType\ImageToImage\ImageToImageInput;
use Drupal\ai\OperationType\ImageToImage\ImageToImageInterface;
use Drupal\ai\OperationType\ImageToImage\ImageToImageOutput;
use Drupal\ai\OperationType\TextToImage\TextToImageInput;
use Drupal\ai\OperationType\TextToImage\TextToImageOutput;
use Drupal\ai\Traits\OperationType\ChatTrait;
use Drupal\ai\Traits\OperationType\ImageToImageTrait;
use Drupal\Component\Serialization\Json;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the 'bytedance' provider.
 */
#[AiProvider(
  id: 'bytedance',
  label: new TranslatableMarkup('ByteDance ModelArk'),
)]
class ByteDanceProvider extends OpenAiBasedProviderClientBase implements ImageToImageInterface {

  use ChatTrait;
  use ImageToImageTrait;

  /**
   * {@inheritdoc}
   *
   * @return array<string>
   *   The supported operation types.
   */
  public function getSupportedOperationTypes(): array {
    return [
      'chat',
      'text_to_image',
      'image_to_image',
    ];
  }

  /**
   * {@inheritdoc}
   *
   * @param string|null $operation_type
   *   The operation type.
   * @param array<mixed> $capabilities
   *   The capabilities.
   *
   * @return array<string, string>
   *   The configured models.
   *
   * @throws \Drupal\ai\Exception\AiSetupFailureException
   */
  public function getConfiguredModels(?string $operation_type = NULL, array $capabilities = []): array {
    $this->loadClient();

    return $this->getModels($operation_type ?? '', $capabilities);
  }

  /**
   * Get the available models.
   *
   * @param string $operation_type
   *   The operation type.
   * @param array<mixed> $capabilities
   *   The capabilities.
   *
   * @return array<string, string>
   *   The available models.
   */
  public function getModels(string $operation_type, array $capabilities): array {
    $models = [];

    // Chat models.
    if ($operation_type === 'chat') {
      $models['skylark-pro-250415']    = 'Skylark Pro (250415)';
      $models['deepseek-v3']           = 'DeepSeek V3';
      $models['kimi-k2-250711']        = 'Kimi K2 (250711)';
      $models['gpt-oss-120b-250805']   = 'GPT OSS 120B (250805)';
      $models['skylark-vision-250515'] = 'Skylark Vision (250515)';
    }

    // Image generation models.
    if ($operation_type === 'text_to_image' || $operation_type === 'image_to_image') {
      $models['seedream-4-0-250828'] = 'Seedream 4.0 (250828)';
      $models['seedream-4-5-251128'] = 'Seedream 4.5 (251128)';
    }

    return $models;
  }

  /**
   * {@inheritdoc}
   *
   * @param string $model_id
   *   The model ID.
   * @param array<string, mixed> $generalConfig
   *   The general configuration.
   *
   * @return array<string, mixed>
   *   The model settings.
   */
  public function getModelSettings(string $model_id, array $generalConfig = []): array {
    // Chat model settings.
    if (in_array(
      $model_id,
      [
        'skylark-pro-250415',
        'deepseek-v3',
        'kimi-k2-250711',
        'gpt-oss-120b-250805',
        'skylark-vision-250515',
      ]
    )
    ) {
      // ByteDance supports standard chat parameters.
      $generalConfig['max_tokens']['default']  = 4096;
      $generalConfig['temperature']['default'] = 1;
      $generalConfig['top_p']['default']       = 0.7;

      // Add ByteDance-specific parameters.
      if (isset($generalConfig['thinking'])) {
        $generalConfig['thinking']['type']        = 'select';
        $generalConfig['thinking']['label']       = 'Thinking Mode';
        $generalConfig['thinking']['description'] = 'Enable deep thinking/reasoning mode';
        $generalConfig['thinking']['default']     = 'disabled';
        $generalConfig['thinking']['constraints'] = [
          'options' => ['enabled', 'disabled'],
        ];
      }

      if (isset($generalConfig['reasoning_effort'])) {
        $generalConfig['reasoning_effort']['type']        = 'select';
        $generalConfig['reasoning_effort']['label']       = 'Reasoning Effort';
        $generalConfig['reasoning_effort']['description'] = 'Level of reasoning effort for models that support it';
        $generalConfig['reasoning_effort']['default']     = 'medium';
        $generalConfig['reasoning_effort']['constraints'] = [
          'options' => ['minimal', 'low', 'medium', 'high'],
        ];
      }
    }

    // Image model settings.
    if ($model_id === 'seedream-4-0-250828' || $model_id === 'seedream-4-5-251128') {
      $generalConfig['size']['default'] = '2048x2048';

      // Remove options constraint to allow custom sizes.
      if (isset($generalConfig['size']['constraints']['options'])) {
        unset($generalConfig['size']['constraints']['options']);
      }

      // Add other supported parameters for Seedream.
      $generalConfig['seed']                        = $generalConfig['seed'] ?? [];
      $generalConfig['watermark']                   = $generalConfig['watermark'] ?? [];
      $generalConfig['sequential_image_generation'] = $generalConfig['sequential_image_generation'] ?? [];
      $generalConfig['optimize_prompt_options']     = $generalConfig['optimize_prompt_options'] ?? [];
    }

    return $generalConfig;
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\ai\Exception\AiSetupFailureException
   */
  protected function loadClient(): void {
    // Set default endpoint for ByteDance ModelArk.
    // This can be overridden by the 'host' config if the user provides one.
    if (empty($this->getConfig()->get('host'))) {
      $this->setEndpoint('https://ark.ap-southeast.bytepluses.com/api/v3');
    }
    else {
      $this->setEndpoint($this->getConfig()->get('host'));
    }

    try {
      parent::loadClient();
    }
    catch (AiSetupFailureException $e) {
      throw new AiSetupFailureException('Failed to initialize ByteDance client: ' . $e->getMessage(), $e->getCode(), $e);
    }
  }

  /**
   * Generate an image from a text prompt.
   *
   * @param string|\Drupal\ai\OperationType\TextToImage\TextToImageInput $input
   *   The input text or TextToImageInput object.
   * @param string $model_id
   *   The model ID.
   * @param array<mixed> $tags
   *   Extra tags.
   *
   * @return \Drupal\ai\OperationType\TextToImage\TextToImageOutput
   *   The generated image.
   *
   * @throws \Drupal\ai\Exception\AiResponseErrorException
   * @throws \Drupal\ai\Exception\AiSetupFailureException
   * @throws \Exception
   */
  public function textToImage(string|TextToImageInput $input, string $model_id, array $tags = []): TextToImageOutput {
    $this->loadClient();
    if ($input instanceof TextToImageInput) {
      $input = $input->getText();
    }

    // Map configuration to API parameters.
    $payload = [
      'model'  => $model_id,
      'prompt' => $input,
      'size'   => $this->configuration['size'] ?? '2048x2048',
      'n'      => (int) ($this->configuration['n'] ?? 1),
    ];

    // Add optional parameters if they are set in configuration.
    $optional_params = ['seed', 'watermark'];
    foreach ($optional_params as $param) {
      if (isset($this->configuration[$param])) {
        $payload[$param] = $this->configuration[$param];
      }
    }

    // Handle response format.
    $payload['response_format'] = 'url';

    // Ensure watermark is boolean.
    if (isset($payload['watermark'])) {
      $payload['watermark'] = (bool) $payload['watermark'];
    }

    // Handle sequential_image_generation.
    if (isset($this->configuration['sequential_image_generation'])) {
      $payload['sequential_image_generation'] = $this->configuration['sequential_image_generation'];
    }

    // Handle optimize_prompt_options
    // The configuration provides the mode as a string (e.g. 'standard')
    // but the API expects an object: {"mode": "standard"}.
    if (isset($this->configuration['optimize_prompt_options'])) {
      $payload['optimize_prompt_options'] = [
        'mode' => $this->configuration['optimize_prompt_options'],
      ];
    }

    try {
      $response = $this->client->images()->create($payload)->toArray();
    }
    catch (\Exception $e) {
      throw $e;
    }

    $images = [];
    if (empty($response['data'][0])) {
      throw new AiResponseErrorException('No image data found in the response.');
    }

    foreach ($response['data'] as $data) {
      if (isset($data['b64_json'])) {
        $images[] = new ImageFile(base64_decode($data['b64_json']), 'image/png', 'seedream.png');
      }
      elseif (isset($data['url'])) {
        $image_content = file_get_contents($data['url']);
        if ($image_content !== FALSE) {
          $images[] = new ImageFile($image_content, 'image/png', 'seedream.png');
        }
      }
    }

    if (empty($images)) {
      throw new AiResponseErrorException('Failed to process any valid images from the API response.');
    }

    // @phpstan-ignore-next-line - ImageFile extends AbstractFileBase, PHPDoc in AI module incorrectly says ImageType
    return new TextToImageOutput($images, $response, []);
  }

  /**
   * {@inheritdoc}
   *
   * @param string|array<mixed>|\Drupal\ai\OperationType\ImageToImage\ImageToImageInput $input
   *   The input image or ImageToImageInput object.
   * @param string $model_id
   *   The model ID.
   * @param array<mixed> $tags
   *   Extra tags.
   *
   * @return \Drupal\ai\OperationType\ImageToImage\ImageToImageOutput
   *   The generated image.
   *
   * @throws \Drupal\ai\Exception\AiRateLimitException
   * @throws \Drupal\ai\Exception\AiRequestErrorException
   * @throws \Drupal\ai\Exception\AiSetupFailureException
   * @throws \Drupal\ai\Exception\AiUnsafePromptException
   */
  public function imageToImage(string|array|ImageToImageInput $input, string $model_id, array $tags = []): ImageToImageOutput {
    $this->loadClient();

    if (!$input instanceof ImageToImageInput) {
      throw new \InvalidArgumentException('Input must be an instance of ImageToImageInput');
    }

    // Get the image.
    $image_file = $input->getImageFile();
    // Try to construct a data URI.
    $image_data = $image_file->getAsBase64EncodedString();

    // If the input has a prompt, use it.
    $prompt = $input->getPrompt();
    if (empty($prompt)) {
      $prompt = 'Generate an image based on the input.';
    }

    // Map configuration to API parameters.
    $payload = [
      'model'  => $model_id,
      'prompt' => $prompt,
      'image'  => $image_data,
      'size'   => $this->configuration['size'] ?? '2048x2048',
      'n'      => (int) ($this->configuration['n'] ?? 1),
    ];

    // Add optional parameters if they are set in configuration.
    $optional_params = ['seed', 'watermark'];
    foreach ($optional_params as $param) {
      if (isset($this->configuration[$param])) {
        $payload[$param] = $this->configuration[$param];
      }
    }

    // Handle sequential_image_generation.
    if (isset($this->configuration['sequential_image_generation'])) {
      $payload['sequential_image_generation'] = $this->configuration['sequential_image_generation'];
    }

    // Handle optimize_prompt_options.
    if (isset($this->configuration['optimize_prompt_options'])) {
      $payload['optimize_prompt_options'] = [
        'mode' => $this->configuration['optimize_prompt_options'],
      ];
    }

    // Handle response format.
    $payload['response_format'] = 'url';

    // Ensure watermark is boolean.
    if (isset($payload['watermark'])) {
      $payload['watermark'] = (bool) $payload['watermark'];
    }

    try {
      $response = $this->client->images()->create($payload)->toArray();

      $images = [];
      foreach ($response['data'] as $item) {
        if (isset($item['url'])) {
          $image_content = file_get_contents($item['url']);
          if ($image_content !== FALSE) {
            $images[] = new ImageFile($image_content, 'image/png', 'seedream.png');
          }
        }
        elseif (isset($item['b64_json'])) {
          $images[] = new ImageFile(base64_decode($item['b64_json']), 'image/png', 'seedream.png');
        }
      }

      // @phpstan-ignore-next-line - ImageFile extends AbstractFileBase, PHPDoc in AI module incorrectly says ImageType
      return new ImageToImageOutput($images, $response, []);
    }
    catch (\Exception $e) {
      if (str_contains($e->getMessage(), 'Sensitive Information')) {
        throw new AiUnsafePromptException('The prompt was flagged as unsafe.', 0, $e);
      }
      if (str_contains($e->getMessage(), 'RPM/TPM Limit Exceeded')) {
        throw new AiRateLimitException('Rate limit exceeded.', 0, $e);
      }
      throw new AiRequestErrorException('Error invoking model response: ' . $e->getMessage(), 0, $e);
    }
  }

  /**
   * {@inheritdoc}
   *
   * @param array<mixed>|string|\Drupal\ai\OperationType\Chat\ChatInput $input
   *   The chat input.
   * @param string $model_id
   *   The model ID.
   * @param array<mixed> $tags
   *   Extra tags.
   *
   * @return \Drupal\ai\OperationType\Chat\ChatOutput
   *   The chat output.
   *
   * @throws \Drupal\ai\Exception\AiQuotaException
   * @throws \Drupal\ai\Exception\AiRateLimitException
   * @throws \Drupal\ai\Exception\AiRequestErrorException
   * @throws \Drupal\ai\Exception\AiSetupFailureException
   * @throws \Drupal\ai\Exception\AiUnsafePromptException
   * @throws \Throwable
   */
  public function chat(array|string|ChatInput $input, string $model_id, array $tags = []): ChatOutput {
    $this->loadClient();

    // Normalize the input if needed.
    $chat_input = $input;
    if ($input instanceof ChatInput) {
      $chat_input = [];
      // Add a system role if wanted.
      if ($this->chatSystemRole) {
        $chat_input[] = [
          'role'    => 'system',
          'content' => $this->chatSystemRole,
        ];
      }

      /**
       * @var \Drupal\ai\OperationType\Chat\ChatMessage $message
       */
      foreach ($input->getMessages() as $message) {
        // For chat, we only support plain text messages with ModelArk.
        $content = [
          [
            'type' => 'text',
            'text' => $message->getText(),
          ],
        ];

        // Handle multimodal content (images).
        if (count($message->getFiles())) {
          foreach ($message->getFiles() as $file) {
            if ($file instanceof ImageFile) {
              // ModelArk expects either an image URL or a data URL.
              // Use a data URL
              // with the correct mime type so the API can validate the payload.
              $data_url  = $file->getAsBase64EncodedString('data:' . $file->getMimeType() . ';base64,');
              $content[] = [
                'type'      => 'image_url',
                'image_url' => [
                  'url'    => $data_url,
                  'detail' => 'auto',
                ],
              ];
            }
          }
        }
        elseif (count($message->getImages())) {
          foreach ($message->getImages() as $image) {
            $data_url  = $image->getAsBase64EncodedString('data:' . $image->getMimeType() . ';base64,');
            $content[] = [
              'type'      => 'image_url',
              'image_url' => [
                'url'    => $data_url,
                'detail' => 'auto',
              ],
            ];
          }
        }

        $new_message = [
          'role'    => $message->getRole(),
          'content' => $content,
        ];

        // If it's a tool's response.
        if ($message->getToolsId()) {
          $new_message['tool_call_id'] = $message->getToolsId();
        }

        // If we want the results from some older tools call.
        if ($message->getTools()) {
          $new_message['tool_calls'] = $message->getRenderedTools();
        }

        $chat_input[] = $new_message;
      }
    }

    $payload = [
      'model'    => $model_id,
      'messages' => $chat_input,
    ] + $this->configuration;

    // Add tools if provided.
    if ($input instanceof ChatInput && $input->getChatTools()) {
      $payload['tools'] = $input->getChatTools()->renderToolsArray();
      foreach ($payload['tools'] as $key => $tool) {
        $payload['tools'][$key]['function']['strict'] = FALSE;
      }
    }

    if ($input instanceof ChatInput) {
      $payload['response_format'] = [
        'type' => 'text',
      ];
    }

    // Handle ByteDance-specific thinking parameter.
    if (isset($this->configuration['thinking'])) {
      $payload['thinking'] = [
        'type' => $this->configuration['thinking'],
      ];
    }

    // Include usage for streamed responses.
    if ($this->streamed) {
      $payload['stream_options']['include_usage'] = TRUE;
    }

    try {
      if ($this->streamed) {
        $response = $this->client->chat()->createStreamed($payload);
        $message  = new OpenAiTypeStreamedChatMessageIterator($response);
      }
      // If we are in a fiber, use streamed response.
      elseif (\Fiber::getCurrent()) {
        $payload['stream_options'] = [
          'include_usage' => TRUE,
        ];
        $response                  = $this->client->chat()
          ->createStreamed($payload);
        $stream                    = new OpenAiTypeStreamedChatMessageIterator($response);
        // Consume the stream in a fiber.
        // Suspend fiber if we haven't finished yet.
        if (empty($stream->getFinishReason())) {
          \Fiber::suspend();
        }

        // Create the final message from accumulated data.
        $message = $stream->reconstructChatOutput()->getNormalized();
      }
      else {
        $response = $this->client->chat()->create($payload)->toArray();
        // If tools are generated.
        $tools = [];
        if (!empty($response['choices'][0]['message']['tool_calls']) && $input instanceof ChatInput) {
          foreach ($response['choices'][0]['message']['tool_calls'] as $tool) {
            $arguments = Json::decode($tool['function']['arguments']);
            $tools[]   = new ToolsFunctionOutput($input->getChatTools()
              ->getFunctionByName($tool['function']['name']), $tool['id'], $arguments);
          }
        }
        $message = new ChatMessage($response['choices'][0]['message']['role'], $response['choices'][0]['message']['content'] ?? '', []);
        if (!empty($tools)) {
          $message->setTools($tools);
        }
      }
    }
    catch (\Exception $e) {
      // Handle ByteDance-specific errors.
      if (str_contains($e->getMessage(), 'Request too large')) {
        throw new AiRateLimitException($e->getMessage());
      }
      if (str_contains($e->getMessage(), 'Too Many Requests')) {
        throw new AiRateLimitException($e->getMessage());
      }
      if (str_contains($e->getMessage(), 'exceeded your current quota')) {
        throw new AiQuotaException($e->getMessage());
      }
      if (str_contains($e->getMessage(), 'Sensitive Information')) {
        throw new AiUnsafePromptException('The prompt was flagged as unsafe.', 0, $e);
      }
      throw new AiRequestErrorException('Error invoking chat response: ' . $e->getMessage(), 0, $e);
    }

    $chat_output = new ChatOutput($message, $response, []);

    // Set token usage if not streamed or in a fiber.
    if (!$this->streamed && !\Fiber::getCurrent()) {
      $this->setChatTokenUsage($chat_output, $response);
    }

    return $chat_output;
  }

  /**
   * {@inheritdoc}
   *
   * @return array<string, mixed>
   *   The setup data.
   */
  public function getSetupData(): array {
    return [
      'key_config_name' => 'api_key',
      'default_models'  => [
        'chat'           => 'skylark-pro-250415',
        'text_to_image' => 'seedream-4-5-251128',
        "image_to_image" => 'seedream-4-5-251128',
      ],
    ];
  }

}
