<?php

declare(strict_types=1);

namespace Drupal\ai_provider_dropsolidai\Form;

use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Extension\ExtensionPathResolver;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\Key\KeyRepositoryInterface;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\BadResponseException;
use GuzzleHttp\Exception\GuzzleException;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Configuration form for Dropsolid AI provider settings.
 */
final class DropsolidAiConfigForm extends ConfigFormBase {

  /**
   * Configuration name used by the form.
   */
  private const CONFIG_NAME = 'ai_provider_dropsolidai.settings';

  /**
   * Cache TTL for test results (1 hour in seconds).
   */
  private const CACHE_TTL = 3600;

  /**
   * Dropsolid marketing URLs.
   */
  private const ABOUT_URL = 'https://dropsolid.ai';
  private const PRICING_URL = 'https://dropsolid.ai/ai-pricing';
  private const AI_SOLUTIONS_URL = 'https://dropsolid.ai/solutions/enhance-your-website-ai-powered-solutions';

  /**
   * Extension path resolver.
   */
  private ExtensionPathResolver $extensionPathResolver;

  /**
   * Module handler service.
   */
  private ModuleHandlerInterface $moduleHandler;

  /**
   * Key repository service.
   */
  private KeyRepositoryInterface $keyRepository;

  /**
   * HTTP client service.
   */
  private Client $httpClient;

  /**
   * LangFuse client service.
   *
   * Optional, only available when langfuse module is installed.
   */
  private ?object $langFuseClient;

  /**
   * AI VDB Provider plugin manager.
   *
   * Optional, only available when ai module is installed.
   */
  private ?object $vdbProviderPluginManager;

  /**
   * Cached LiteLLM configuration object for performance optimization.
   */
  private ?ImmutableConfig $liteLlmConfig;

  /**
   * Cached Postgres VDB configuration object for performance optimization.
   */
  private ?ImmutableConfig $postgresVdbConfig;

  /**
   * Cached LangFuse configuration object for performance optimization.
   */
  private ?ImmutableConfig $langFuseConfig;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    /** @var static $instance */
    $instance = parent::create($container);
    $instance->moduleHandler = $container->get('module_handler');
    $instance->extensionPathResolver = $container->get('extension.path.resolver');
    $instance->keyRepository = $container->get('key.repository');
    $instance->httpClient = $container->get('http_client');

    // VDB Provider plugin manager is optional - only inject if service exists.
    if ($container->has('ai.vdb_provider')) {
      $instance->vdbProviderPluginManager = $container->get('ai.vdb_provider');
    }
    else {
      $instance->vdbProviderPluginManager = NULL;
    }

    // LangFuse client is optional - only inject if the service exists.
    if ($container->has('langfuse.client')) {
      $instance->langFuseClient = $container->get('langfuse.client');
    }
    else {
      $instance->langFuseClient = NULL;
    }

    // Cache configuration objects for performance optimization.
    $instance->liteLlmConfig = $container->get('config.factory')->get('ai_provider_litellm.settings');
    $instance->postgresVdbConfig = $container->get('config.factory')->get('ai_vdb_provider_postgres.settings');
    $instance->langFuseConfig = $container->get('config.factory')->get('langfuse.settings');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ai_provider_dropsolidai_config_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [self::CONFIG_NAME];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['header'] = $this->buildHeader();

    // Add a helpful note about route availability.
    $form['route_note'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['dsai-route-note']],
      'message' => [
        '#markup' => '<div class="messages messages--warning"><strong>' . $this->t('Note:') . '</strong> ' . $this->t('This form provides guidance for setting up AI components. After installing modules, you may need to navigate to the specific admin sections mentioned below to complete configuration.') . '</div>',
      ],
    ];

    $form['tabs'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Setup steps'),
    ];

    // Automatic Setup (Dropsolid Platform SSO)
    $form['auto_setup'] = $this->buildAutoSetupSection();

    // Manual Setup.
    $form['manual_setup'] = [
      '#type' => 'details',
      '#title' => $this->t('Manual Setup'),
      '#group' => 'tabs',
      '#open' => TRUE,
      '#description' => $this->t('Configure each AI component manually using dedicated settings forms. After installing modules, navigate to the respective admin configuration sections to set them up.'),
    ];

    // Provider AI LiteLLM Setup.
    $form['manual_setup']['provider'] = $this->buildProviderSection();

    // Vector Store (Optional)
    $form['manual_setup']['vector_store'] = $this->buildVectorStoreSection();

    // Tracing & Analytics - Langfuse (Optional)
    $form['manual_setup']['tracing'] = $this->buildTracingSection();

    // AI Extras - AI Dropsolid (Optional)
    $form['manual_setup']['ai_extras'] = $this->buildAiExtraSection();

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->configFactory()->getEditable(self::CONFIG_NAME);

    // Save current automatic checkbox states (not form state values since
    // checkboxes are disabled).
    $config
      ->set('provider.configured', $this->isLiteLlmConfigured())
      ->set('provider.tested', $this->isLiteLlmTested())
      ->set('vdb.configured', $this->isPostgresVdbConfigured())
      ->set('vdb.tested', $this->isPostgresVdbTested())
      ->set('tracing.configured', $this->isLangFuseConfigured())
      ->set('tracing.tested', $this->isLangFuseTested());

    $config->save();

    $this->messenger()->addStatus($this->t('Dropsolid AI provider configuration has been saved.'));

    parent::submitForm($form, $form_state);
  }

  /**
   * AJAX callback for provider self-test.
   *
   * Tests the LiteLLM provider connection by attempting to retrieve available
   * models. This method performs comprehensive validation including module
   * installation check, configuration validation, API key verification, and
   * actual connection testing. Results are cached for performance optimization
   * and displayed with user-friendly error messages while logging detailed
   * exceptions for debugging.
   *
   * @param array &$form
   *   The form array structure. Modified to update test results and button
   *   states.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form. Used to maintain form state during AJAX
   *   operations.
   *
   * @return array
   *   The form element containing the test results to be updated via AJAX.
   *
   * @throws \Exception
   *   Propagates exceptions that are caught and handled with user-friendly
   *   messages. Full exception details are logged for debugging purposes.
   */
  public function testProvider(array &$form, FormStateInterface $form_state): array {
    // Get reference to test container to reduce repetitive array indexing.
    $test = &$form['manual_setup']['provider']['test_container'];

    // Disable the test button during testing.
    $this->setTestButtonState($test, '⏳ Testing...', TRUE);

    // Check if LiteLLM module is installed.
    if (!$this->moduleHandler->moduleExists('ai_provider_litellm')) {
      $test['test_results'] = $this->buildErrorMessage($this->createModuleNotInstalledError('LiteLLM', 'ai_provider_litellm'));
      // Update checkbox to reflect failed test.
      $test['test_litellm']['#value'] = FALSE;
      return $this->finalizeAjax($test, '🔍 Test Connection', $form_state);
    }

    // Get LiteLLM configuration.
    $api_key_id = $this->liteLlmConfig->get('api_key');
    $host = $this->liteLlmConfig->get('host');

    // Check if API key is configured.
    if (empty($api_key_id)) {
      $test['test_results'] = $this->buildErrorMessage((string) $this->t('❌ LiteLLM API key is not configured. Please configure it in the LiteLLM settings.'));
      // Update checkbox to reflect failed test.
      $test['test_litellm']['#value'] = FALSE;
      return $this->finalizeAjax($test, '🔍 Test Connection', $form_state);
    }

    // Check if host is configured.
    if (empty($host)) {
      $test['test_results'] = $this->buildErrorMessage((string) $this->t('❌ LiteLLM host is not configured. Please configure it in the LiteLLM settings.'));
      // Update checkbox to reflect failed test.
      $test['test_litellm']['#value'] = FALSE;
      return $this->finalizeAjax($test, '🔍 Test Connection', $form_state);
    }

    try {
      // Get the actual API key value.
      $api_key = $this->keyRepository->getKey($api_key_id);
      if (!$api_key) {
        $test['test_results'] = $this->buildErrorMessage((string) $this->t('❌ LiteLLM API key is invalid or not found.'));
        // Update checkbox to reflect failed test.
        $test['test_litellm']['#value'] = FALSE;
        return $this->finalizeAjax($test, '🔍 Test Connection', $form_state);
      }

      $api_key_value = $api_key->getKeyValue();
      if (empty($api_key_value)) {
        $test['test_results'] = $this->buildErrorMessage((string) $this->t('❌ LiteLLM API key value is empty.'));
        // Update checkbox to reflect failed test.
        $test['test_litellm']['#value'] = FALSE;
        return $this->finalizeAjax($test, '🔍 Test Connection', $form_state);
      }

      // Try to create the LiteLLM client and test connection.
      // We need to use reflection or check if the class exists since we can't
      // directly import it.
      if (!class_exists('\Drupal\ai_provider_litellm\LiteLLM\LiteLlmAiClient')) {
        $test['test_results'] = $this->buildErrorMessage((string) $this->t('❌ LiteLLM client class not found. The module may not be properly configured - check the configurations.'));
        // Update checkbox to reflect failed test.
        $test['test_litellm']['#value'] = FALSE;
        return $this->finalizeAjax($test, '🔍 Test Connection', $form_state);
      }

      $client_class = '\Drupal\ai_provider_litellm\LiteLLM\LiteLlmAiClient';
      $client = new $client_class(
        $this->httpClient,
        $this->keyRepository,
        $host,
        $api_key_id,
        $this->liteLlmConfig->get('moderation') ?? TRUE
      );

      // Test the connection by trying to get models.
      $models = $client->models();
      if (empty($models)) {
        $test['test_results'] = $this->buildWarningMessage((string) $this->t('⚠️ Connected to LiteLLM but no models are available. Check your LiteLLM configuration.'));
        return $this->finalizeAjax($test, '🔍 Test Connection', $form_state);
      }

      // Success!
      $model_count = count($models);
      $test['test_results'] = $this->buildSuccessMessage((string) $this->t('✅ LiteLLM connection successful! Found @count models.', ['@count' => $model_count]));

      // Cache successful test result.
      $this->cacheTestResult(TRUE);

      // Update the test checkbox to reflect successful test.
      $test['test_litellm']['#value'] = TRUE;
      $test['test_litellm']['#default_value'] = TRUE;
      $test['test_litellm']['#checked'] = TRUE;
      return $this->finalizeAjax($test, '🔍 Test Connection', $form_state);
    }
    catch (BadResponseException $e) {
      // Log the full exception for debugging.
      $this->loggerFactory->get('ai_provider_dropsolidai')
        ->error('LiteLLM connection failed with HTTP error: @message. Response: @response', [
          '@message' => $e->getMessage(),
          '@response' => $e->getResponse() ? $e->getResponse()->getBody()->getContents() : 'No response body',
        ]);

      $error_message = $this->t('❌ LiteLLM connection failed: @message', ['@message' => $e->getMessage()]);

      // Handle specific error cases.
      if ($e->getCode() === Response::HTTP_UNAUTHORIZED) {
        $error_message = $this->t('❌ LiteLLM authentication failed. Please check your API key.');
      }
      elseif ($e->getCode() === Response::HTTP_BAD_REQUEST) {
        try {
          $response_body = json_decode($e->getResponse()->getBody()->getContents(), TRUE);
          if (isset($response_body['error']['type']) && $response_body['error']['type'] === 'budget_exceeded') {
            $error_message = $this->t('❌ LiteLLM budget exceeded: @message', ['@message' => $response_body['error']['message'] ?? 'Unknown budget error']);
          }
        }
        catch (\Exception $parse_exception) {
          // Ignore JSON parsing errors.
        }
      }
      elseif ($e->getCode() === Response::HTTP_INTERNAL_SERVER_ERROR) {
        try {
          $response_body = json_decode($e->getResponse()->getBody()->getContents(), TRUE);
          if (isset($response_body['detail']['error']) && str_starts_with($response_body['detail']['error'], 'LLM Model List not loaded')) {
            $error_message = $this->t('⚠️ LiteLLM Model List not loaded - unable to retrieve dynamic model list.');
          }
        }
        catch (\Exception $parse_exception) {
          // Ignore JSON parsing errors.
        }
      }

      $test['test_results'] = $this->buildErrorMessage((string) $error_message);
      // Update checkbox to reflect failed test.
      $test['test_litellm']['#value'] = FALSE;
      return $this->finalizeAjax($test, '🔍 Test Connection', $form_state);
    }
    catch (GuzzleException $e) {
      // Log the full exception for debugging.
      $this->loggerFactory->get('ai_provider_dropsolidai')
        ->error('LiteLLM connection failed with network error: @message', [
          '@message' => $e->getMessage(),
        ]);

      $test['test_results'] = $this->buildErrorMessage($this->createConnectionFailedError('LiteLLM', $e->getMessage()));
      // Update checkbox to reflect failed test.
      $test['test_litellm']['#value'] = FALSE;
      return $this->finalizeAjax($test, '🔍 Test Connection', $form_state);
    }
    catch (\Exception $e) {
      // Log the full exception for debugging.
      $this->loggerFactory->get('ai_provider_dropsolidai')
        ->error('Unexpected error testing LiteLLM: @message. Stack trace: @trace', [
          '@message' => $e->getMessage(),
          '@trace' => $e->getTraceAsString(),
        ]);

      $test['test_results'] = $this->buildErrorMessage((string) $this->t('❌ Unexpected error testing LiteLLM: @message', ['@message' => $e->getMessage()]));
      // Update checkbox to reflect failed test.
      $test['test_litellm']['#value'] = FALSE;
      return $this->finalizeAjax($test, '🔍 Test Connection', $form_state);
    }
  }

  /**
   * AJAX callback for vector store checks.
   *
   * Tests the Postgres vector database provider connection by attempting to
   * establish a connection and perform a ping operation. This method validates
   * module installation, configuration completeness, password key validity,
   * and actual database connectivity. Results are cached for performance and
   * displayed with user-friendly error messages while logging detailed
   * exceptions for debugging.
   *
   * @param array &$form
   *   The form array structure. Modified to update test results and button
   *   states.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form. Used to maintain form state during AJAX
   *   operations.
   *
   * @return array
   *   The form element containing the test results to be updated via AJAX.
   *
   * @throws \Exception
   *   Propagates exceptions that are caught and handled with user-friendly
   *   messages. Full exception details are logged for debugging purposes.
   */
  public function testVectorStore(array &$form, FormStateInterface $form_state): array {
    // Get reference to test container to reduce repetitive array indexing.
    $test = &$form['manual_setup']['vector_store']['test_container'];

    // Disable the test button during testing.
    $this->setTestButtonState($test, '⏳ Testing...', TRUE);

    // Check if Postgres VDB module is installed.
    if (!$this->moduleHandler->moduleExists('ai_vdb_provider_postgres')) {
      $test['test_results'] = $this->buildErrorMessage($this->createModuleNotInstalledError('Postgres VDB', 'ai_vdb_provider_postgres'));
      // Update checkbox to reflect failed test.
      $test['test_vdb']['#value'] = FALSE;
      return $this->finalizeAjax($test, '🔍 Test Vector Store', $form_state);
    }

    // Get Postgres VDB configuration.
    $host = $this->postgresVdbConfig->get('host');
    $username = $this->postgresVdbConfig->get('username');
    $password_key_id = $this->postgresVdbConfig->get('password');
    $database = $this->postgresVdbConfig->get('default_database');

    // Check if required settings are configured.
    if (empty($host) || empty($username) || empty($password_key_id) || empty($database)) {
      $test['test_results'] = $this->buildErrorMessage($this->createConfigurationMissingError('Postgres VDB'));
      // Update checkbox to reflect failed test.
      $test['test_vdb']['#value'] = FALSE;
      return $this->finalizeAjax($test, '🔍 Test Vector Store', $form_state);
    }

    try {
      // Get the actual password value.
      $password_key = $this->keyRepository->getKey($password_key_id);
      if (!$password_key) {
        $test['test_results'] = $this->buildErrorMessage((string) $this->t('❌ Postgres VDB password key is invalid or not found.'));
        // Update checkbox to reflect failed test.
        $test['test_vdb']['#value'] = FALSE;
        return $this->finalizeAjax($test, '🔍 Test Vector Store', $form_state);
      }

      $password = $password_key->getKeyValue();
      if (empty($password)) {
        $test['test_results'] = $this->buildErrorMessage((string) $this->t('❌ Postgres VDB password value is empty.'));
        // Update checkbox to reflect failed test.
        $test['test_vdb']['#value'] = FALSE;
        return $this->finalizeAjax($test, '🔍 Test Vector Store', $form_state);
      }

      // Try to create the Postgres VDB connector and test connection.
      if (!$this->vdbProviderPluginManager) {
        $test['test_results'] = $this->buildErrorMessage((string) $this->t('❌ AI VDB Provider plugin manager is not available. Please ensure the AI module is installed and enabled.'));
        // Update checkbox to reflect failed test.
        $test['test_vdb']['#value'] = FALSE;
        return $this->finalizeAjax($test, '🔍 Test Vector Store', $form_state);
      }

      $connector = $this->vdbProviderPluginManager->createInstance('postgres');

      // Set custom configuration.
      $port = $this->postgresVdbConfig->get('port') ?: '5432';
      $connector->setCustomConfig([
        'host' => $host,
        'port' => $port,
        'username' => $username,
        'password' => $password,
        'default_database' => $database,
      ]);

      // Test the connection by trying to ping.
      if (!$connector->ping()) {
        $test['test_results'] = $this->buildErrorMessage($this->createConnectionFailedError('Postgres VDB', 'Please check your database settings.'));
        return $this->finalizeAjax($test, '🔍 Test Vector Store', $form_state);
      }

      // Success!
      $test['test_results'] = $this->buildSuccessMessage($this->createConnectionSuccessMessage('Postgres VDB'));

      // Cache successful test result.
      $this->cacheVdbTestResult(TRUE);

      // Update the test checkbox to reflect successful test.
      $test['test_vdb']['#value'] = TRUE;
      $test['test_vdb']['#default_value'] = TRUE;
      $test['test_vdb']['#checked'] = TRUE;
      return $this->finalizeAjax($test, '🔍 Test Vector Store', $form_state);
    }
    catch (\Exception $e) {
      // Log the full exception for debugging.
      $this->loggerFactory->get('ai_provider_dropsolidai')
        ->error('Unexpected error testing Postgres VDB: @message. Stack trace: @trace', [
          '@message' => $e->getMessage(),
          '@trace' => $e->getTraceAsString(),
        ]);

      $test['test_results'] = $this->buildErrorMessage((string) $this->t('❌ Unexpected error testing Postgres VDB: @message', ['@message' => $e->getMessage()]));
      // Update checkbox to reflect failed test.
      $test['test_vdb']['#value'] = FALSE;
      return $this->finalizeAjax($test, '🔍 Test Vector Store', $form_state);
    }
  }

  /**
   * AJAX callback for tracing tests.
   *
   * Tests the LangFuse tracing and analytics service connection by validating
   * module installation, configuration completeness, authentication method
   * setup, and actual service connectivity. This method handles multiple
   * authentication methods (key_pair, bearer, basic, none) and performs
   * comprehensive validation before attempting connection. Results are cached
   * for performance and displayed with user-friendly error messages while
   * logging detailed exceptions for debugging.
   *
   * @param array &$form
   *   The form array structure. Modified to update test results and button
   *   states.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form. Used to maintain form state during AJAX
   *   operations.
   *
   * @return array
   *   The form element containing the test results to be updated via AJAX.
   *
   * @throws \Exception
   *   Propagates exceptions that are caught and handled with user-friendly
   *   messages. Full exception details are logged for debugging purposes.
   */
  public function testTracing(array &$form, FormStateInterface $form_state): array {
    // Get reference to test container to reduce repetitive array indexing.
    $test = &$form['manual_setup']['tracing']['test_container'];

    // Disable the test button during testing.
    $this->setTestButtonState($test, '⏳ Testing...', TRUE);

    // Check if LangFuse module is installed.
    if (!$this->moduleHandler->moduleExists('langfuse')) {
      $test['test_results'] = $this->buildErrorMessage($this->createModuleNotInstalledError('LangFuse', 'langfuse'));
      // Update checkbox to reflect failed test.
      $test['test_tracing']['#value'] = FALSE;
      return $this->finalizeAjax($test, '🔍 Test Tracing', $form_state);
    }

    // Check if LangFuse AI Logging sub-module is installed.
    if (!$this->moduleHandler->moduleExists('langfuse_ai_logging')) {
      $test['test_results'] = $this->buildErrorMessage((string) $this->t('❌ LangFuse AI Logging module is not enabled. Please enable both langfuse and langfuse_ai_logging modules.'));
      // Update checkbox to reflect failed test.
      $test['test_tracing']['#value'] = FALSE;
      return $this->finalizeAjax($test, '🔍 Test Tracing', $form_state);
    }

    // Get LangFuse configuration.
    $langfuse_url = $this->langFuseConfig->get('langfuse_url');
    $auth_method = $this->langFuseConfig->get('auth_method');

    // Check if required settings are configured.
    if (empty($langfuse_url) || empty($auth_method)) {
      $test['test_results'] = $this->buildErrorMessage($this->createConfigurationMissingError('LangFuse'));
      // Update checkbox to reflect failed test.
      $test['test_tracing']['#value'] = FALSE;
      return $this->finalizeAjax($test, '🔍 Test Tracing', $form_state);
    }

    try {
      // Check if LangFuse client class exists.
      if (!class_exists('\Drupal\langfuse\LangFuseClient')) {
        $test['test_results'] = $this->buildErrorMessage((string) $this->t('❌ LangFuse client class not found. The module may not be properly installed.'));
        // Update checkbox to reflect failed test.
        $test['test_tracing']['#value'] = FALSE;
        return $this->finalizeAjax($test, '🔍 Test Tracing', $form_state);
      }

      // Build configuration array for connection test.
      $config = ['host' => $langfuse_url];

      switch ($auth_method) {
        case 'key_pair':
          $key_pair = $this->langFuseConfig->get('key_pair');
          if (empty($key_pair['public_key']) || empty($key_pair['secret_key'])) {
            $test['test_results'] = $this->buildErrorMessage((string) $this->t('❌ LangFuse key pair is not configured.'));
            // Update checkbox to reflect failed test.
            $test['test_tracing']['#value'] = FALSE;
            return $this->finalizeAjax($test, '🔍 Test Tracing', $form_state);
          }
          $config['public_key'] = $key_pair['public_key'];
          $config['secret_key'] = $key_pair['secret_key'];
          break;

        case 'bearer':
          $bearer_token = $this->langFuseConfig->get('bearer_token');
          if (empty($bearer_token)) {
            $test['test_results'] = $this->buildErrorMessage((string) $this->t('❌ LangFuse bearer token is not configured.'));
            // Update checkbox to reflect failed test.
            $test['test_tracing']['#value'] = FALSE;
            return $this->finalizeAjax($test, '🔍 Test Tracing', $form_state);
          }
          $config['public_key'] = $bearer_token;
          $config['secret_key'] = $bearer_token;
          break;

        case 'basic':
          $basic_auth = $this->langFuseConfig->get('basic_auth');
          if (empty($basic_auth['username']) || empty($basic_auth['password'])) {
            $test['test_results'] = $this->buildErrorMessage((string) $this->t('❌ LangFuse basic authentication is not configured.'));
            // Update checkbox to reflect failed test.
            $test['test_tracing']['#value'] = FALSE;
            return $this->finalizeAjax($test, '🔍 Test Tracing', $form_state);
          }
          $config['public_key'] = $basic_auth['username'];
          $config['secret_key'] = $basic_auth['password'];
          break;

        case 'none':
          // No authentication required.
          break;

        default:
          $test['test_results'] = $this->buildErrorMessage((string) $this->t('❌ Unsupported authentication method: @method', ['@method' => $auth_method]));
          // Update checkbox to reflect failed test.
          $test['test_tracing']['#value'] = FALSE;
          return $this->finalizeAjax($test, '🔍 Test Tracing', $form_state);
      }

      // Try to get the LangFuse client service and test connection.
      if (!$this->langFuseClient) {
        $test['test_results'] = $this->buildErrorMessage((string) $this->t('❌ LangFuse client service not available.'));
        // Update checkbox to reflect failed test.
        $test['test_tracing']['#value'] = FALSE;
        return $this->finalizeAjax($test, '🔍 Test Tracing', $form_state);
      }

      // Test the connection.
      if (!$this->langFuseClient->checkConnection($config)) {
        $test['test_results'] = $this->buildErrorMessage($this->createConnectionFailedError('LangFuse', 'Please check your settings.'));
        return $this->finalizeAjax($test, '🔍 Test Tracing', $form_state);
      }

      // Success!
      $test['test_results'] = $this->buildSuccessMessage($this->createConnectionSuccessMessage('LangFuse'));

      // Cache successful test result.
      $this->cacheLangFuseTestResult(TRUE);

      // Update the test checkbox to reflect successful test.
      $test['test_tracing']['#value'] = TRUE;
      $test['test_tracing']['#default_value'] = TRUE;
      $test['test_tracing']['#checked'] = TRUE;
      return $this->finalizeAjax($test, '🔍 Test Tracing', $form_state);
    }
    catch (\Exception $e) {
      // Log the full exception for debugging.
      $this->loggerFactory->get('ai_provider_dropsolidai')
        ->error('Unexpected error testing LangFuse: @message. Stack trace: @trace', [
          '@message' => $e->getMessage(),
          '@trace' => $e->getTraceAsString(),
        ]);

      $test['test_results'] = $this->buildErrorMessage((string) $this->t('❌ Unexpected error testing LangFuse: @message', ['@message' => $e->getMessage()]));
      // Update checkbox to reflect failed test.
      $test['test_tracing']['#value'] = FALSE;
      return $this->finalizeAjax($test, '🔍 Test Tracing', $form_state);
    }
  }

  /**
   * No-op submit handler for AJAX buttons.
   */
  public function noopSubmitHandler(array &$form, FormStateInterface $form_state): void {
    // Nothing to do for placeholder submit handler.
  }

  /**
   * Sets the test button state (disabled/enabled) and label.
   *
   * @param array &$container
   *   The test container array containing the test_button element.
   * @param string $label
   *   The button label to display.
   * @param bool $disabled
   *   Whether the button should be disabled.
   */
  private function setTestButtonState(array &$container, string $label, bool $disabled): void {
    $container['test_button']['#disabled'] = $disabled;
    $container['test_button']['#value'] = $label;
  }

  /**
   * Finalizes AJAX test response by ensuring button is re-enabled.
   *
   * Ensures button is re-enabled and form is rebuilt.
   *
   * @param array &$container
   *   The test container array containing the test_button element.
   * @param string $resetLabel
   *   The label to reset the button to.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state to trigger rebuild.
   *
   * @return array
   *   The container array for return.
   */
  private function finalizeAjax(array &$container, string $resetLabel, FormStateInterface $form_state): array {
    $this->setTestButtonState($container, $resetLabel, FALSE);
    $form_state->setRebuild(TRUE);
    return $container;
  }

  /**
   * Builds the header container for the configuration form.
   */
  private function buildHeader(): array {
    $modulePath = $this->extensionPathResolver->getPath('module', 'ai_provider_dropsolidai');
    $logoUrl = Url::fromUri('base:' . $modulePath . '/img/logo.svg')->toString(TRUE)->getGeneratedUrl();

    $links = [
      Link::fromTextAndUrl(
        $this->t('About'),
        Url::fromUri(self::ABOUT_URL, [
          'attributes' => [
            'target' => '_blank',
            'rel' => 'noopener',
          ],
        ])
      )->toRenderable(),
      Link::fromTextAndUrl(
        $this->t('Pricing'),
        Url::fromUri(self::PRICING_URL, [
          'attributes' => [
            'target' => '_blank',
            'rel' => 'noopener',
          ],
        ])
      )->toRenderable(),
      Link::fromTextAndUrl(
        $this->t('AI Solutions'),
        Url::fromUri(self::AI_SOLUTIONS_URL, [
          'attributes' => [
            'target' => '_blank',
            'rel' => 'noopener',
          ],
        ])
      )->toRenderable(),
    ];

    return [
      '#type' => 'container',
      '#attributes' => ['class' => ['dsai-header']],
      'branding' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['dsai-header__branding']],
        'logo' => [
          '#type' => 'html_tag',
          '#tag' => 'img',
          '#attributes' => [
            'src' => $logoUrl,
            'alt' => $this->t('Dropsolid.ai'),
            'class' => ['dsai-header__logo'],
            'style' => 'max-width: 400px; height: auto;',
          ],
        ],
        'title' => [
          '#type' => 'html_tag',
          '#tag' => 'h2',
          '#value' => $this->t('Dropsolid.ai'),
        ],
        'summary' => [
          '#markup' => '<p>' . $this->t('Connect your Drupal site to Dropsolid’s AI control plane, configure managed LLM access, vector storage, and telemetry in a guided workflow.') . '</p>',
        ],
      ],
      'links' => [
        '#theme' => 'item_list',
        '#attributes' => ['class' => ['dsai-header__links']],
        '#items' => $links,
      ],
    ];
  }

  /**
   * Builds the auto setup section (SSO coming soon).
   */
  private function buildAutoSetupSection(): array {
    return [
      '#type' => 'details',
      '#title' => $this->t('Automatic Setup (SSO)'),
      '#group' => 'tabs',
      '#open' => TRUE,
      '#description' => $this->t('Sign in with your Dropsolid account to automatically configure all AI services.'),
      'coming_soon' => [
        '#type' => 'container',
        'description' => [
          '#markup' => '<p><em>' . $this->t('Coming soon: sign in with Dropsolid to auto-configure your provider.') . '</em></p>',
        ],
        'login_button' => [
          '#type' => 'submit',
          '#value' => $this->t('Sign in with Dropsolid'),
          '#disabled' => TRUE,
          '#limit_validation_errors' => [],
          '#submit' => ['::noopSubmitHandler'],
        ],
      ],
    ];
  }

  /**
   * Builds the provider section (LiteLLM setup).
   */
  private function buildProviderSection(): array {
    $hasLitellm = $this->moduleHandler->moduleExists('ai_provider_litellm');

    $section = [
      '#type' => 'details',
      '#title' => $this->t('1. Provider AI LiteLLM Setup'),
      '#open' => TRUE,
      '#description' => $this->t('Configure LiteLLM as your AI provider to access multiple LLM services through a unified API.'),
      'install_litellm' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Install LiteLLM AI Provider module'),
        '#default_value' => $hasLitellm,
        '#description' => $this->t('Download and enable the LiteLLM module to get started.'),
        '#disabled' => TRUE,
      ],
      'configure_litellm' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Configure LiteLLM connection'),
        '#default_value' => $this->isLiteLlmConfigured(),
        '#description' => $this->t('Set up your API keys and connection settings.'),
        '#disabled' => TRUE,
      ],
    ];

    // Add configure links immediately after configure checkbox when module
    // is installed.
    if ($hasLitellm) {
      $section['configure_links'] = [
        '#type' => 'container',
        '#weight' => 15,
        'settings_link' => [
          '#type' => 'link',
          '#title' => $this->t('⚙️ Configure LiteLLM Settings'),
          '#url' => Url::fromRoute('ai_provider_litellm.settings_form'),
          '#attributes' => ['class' => ['button', 'button--primary']],
        ],
        'note' => [
          '#markup' => '<p><em>' . $this->t('Note: Navigate to AI Providers configuration to set up LiteLLM.') . '</em></p>',
        ],
      ];
    }

    // Continue with test elements.
    $section += [
      'test_container' => [
        '#type' => 'container',
        '#attributes' => ['id' => 'ai-provider-dropsolidai-test-container'],
        '#weight' => 20,
        'test_litellm' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Test LiteLLM connection'),
          '#default_value' => $this->isLiteLlmTested(),
          '#description' => $this->t('Verify your connection is working properly.'),
          '#disabled' => TRUE,
          '#weight' => 10,
        ],
        'test_button' => [
          '#type' => 'submit',
          '#value' => $this->t('🔍 Test Connection'),
          '#ajax' => [
            'callback' => '::testProvider',
            'wrapper' => 'ai-provider-dropsolidai-test-container',
            'progress' => [
              'type' => 'throbber',
              'message' => $this->t('Testing LiteLLM connection...'),
            ],
          ],
          '#limit_validation_errors' => [],
          '#submit' => ['::noopSubmitHandler'],
          '#weight' => 20,
        ],
        'test_results' => [
          '#type' => 'container',
          '#attributes' => ['id' => 'ai-provider-dropsolidai-results'],
          '#markup' => '<div class="dsai-results"><em>' . $this->t('Test results will appear here.') . '</em></div>',
          '#weight' => 30,
        ],
      ],
    ];

    // Add install links when module is not installed.
    if (!$hasLitellm) {
      $section['install_links'] = [
        '#type' => 'container',
        '#weight' => 50,
        'download_link' => [
          '#type' => 'link',
          '#title' => $this->t('📥 Download LiteLLM Module'),
          '#url' => Url::fromUri('https://www.drupal.org/project/ai_provider_litellm'),
          '#attributes' => ['target' => '_blank', 'rel' => 'noopener'],
        ],
        'instructions' => [
          '#type' => 'details',
          '#title' => $this->t('Composer & Drush Instructions'),
          '#open' => FALSE,
          '#description' => $this->t('Run these commands to install and enable the module.'),
          'commands' => [
            '#markup' => '<div class="dsai-commands"><code>composer require drupal/ai_provider_litellm</code><br><code>drush en ai_provider_litellm -y</code></div>',
          ],
        ],
      ];
    }

    return $section;
  }

  /**
   * Builds the vector store section (Postgres VDB setup).
   */
  private function buildVectorStoreSection(): array {
    $hasVdb = $this->moduleHandler->moduleExists('ai_vdb_provider_postgres');

    $section = [
      '#type' => 'details',
      '#title' => $this->t('2. Vector Store (Optional)'),
      '#open' => FALSE,
      '#description' => $this->t('Enable vector storage for semantic search and AI-powered content retrieval.'),
      'install_vdb' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Install Postgres VDB Provider module'),
        '#default_value' => $hasVdb,
        '#description' => $this->t('Download and enable the PostgreSQL vector database provider.'),
        '#disabled' => TRUE,
      ],
      'configure_vdb' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Configure vector database connection'),
        '#default_value' => $this->isPostgresVdbConfigured(),
        '#description' => $this->t('Set up your PostgreSQL connection and vector settings.'),
        '#disabled' => TRUE,
      ],
    ];

    // Add configure links immediately after configure checkbox when module
    // is installed.
    if ($hasVdb) {
      $section['configure_links'] = [
        '#type' => 'container',
        '#weight' => 15,
        'settings_link' => [
          '#type' => 'link',
          '#title' => $this->t('⚙️ Configure Vector Database'),
          '#url' => Url::fromRoute('ai_vdb_provider_postgres.settings_form'),
          '#attributes' => ['class' => ['button', 'button--primary']],
        ],
        'note' => [
          '#markup' => '<p><em>' . $this->t('Note: Navigate to AI VDB configuration to set up PostgreSQL vector storage.') . '</em></p>',
        ],
      ];
    }

    // Continue with test elements.
    $section += [
      'test_container' => [
        '#type' => 'container',
        '#attributes' => ['id' => 'ai-provider-dropsolidai-vector-test-container'],
        '#weight' => 20,
        'test_vdb' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Test vector database connection'),
          '#default_value' => $this->isPostgresVdbTested(),
          '#description' => $this->t('Verify your vector database is working properly.'),
          '#disabled' => TRUE,
          '#weight' => 10,
          '#states' => [
            'visible' => [
              ':input[name="manual_setup[vector_store][install_vdb]"]' => ['checked' => TRUE],
              ':input[name="manual_setup[vector_store][configure_vdb]"]' => ['checked' => TRUE],
            ],
          ],
        ],
        'test_button' => [
          '#type' => 'submit',
          '#value' => $this->t('🔍 Test Vector Store'),
          '#ajax' => [
            'callback' => '::testVectorStore',
            'wrapper' => 'ai-provider-dropsolidai-vector-test-container',
            'progress' => [
              'type' => 'throbber',
              'message' => $this->t('Testing vector database connection...'),
            ],
          ],
          '#limit_validation_errors' => [],
          '#submit' => ['::noopSubmitHandler'],
          '#weight' => 20,
          '#states' => [
            'visible' => [
              ':input[name="manual_setup[vector_store][install_vdb]"]' => ['checked' => TRUE],
              ':input[name="manual_setup[vector_store][configure_vdb]"]' => ['checked' => TRUE],
            ],
          ],
        ],
        'test_results' => [
          '#type' => 'container',
          '#attributes' => ['id' => 'ai-provider-dropsolidai-vector-results'],
          '#markup' => '<div class="dsai-results"><em>' . $this->t('Test results will appear here.') . '</em></div>',
          '#weight' => 30,
        ],
      ],
    ];

    // Add install links when module is not installed.
    if (!$hasVdb) {
      $section['install_links'] = [
        '#type' => 'container',
        '#weight' => 50,
        'download_link' => [
          '#type' => 'link',
          '#title' => $this->t('📥 Download VDB Provider Module'),
          '#url' => Url::fromUri('https://www.drupal.org/project/ai_vdb_provider_postgres'),
          '#attributes' => ['target' => '_blank', 'rel' => 'noopener'],
        ],
        'instructions' => [
          '#type' => 'details',
          '#title' => $this->t('Composer & Drush Instructions'),
          '#open' => FALSE,
          '#description' => $this->t('Run these commands to install and enable the module.'),
          'commands' => [
            '#markup' => '<div class="dsai-commands"><code>composer require drupal/ai_vdb_provider_postgres</code><br><code>drush en ai_vdb_provider_postgres -y</code></div>',
          ],
        ],
      ];
    }

    return $section;
  }

  /**
   * Builds the tracing section (LangFuse setup).
   */
  private function buildTracingSection(): array {
    $hasTracing = $this->moduleHandler->moduleExists('langfuse');

    $section = [
      '#type' => 'details',
      '#title' => $this->t('3. Tracing & Analytics - Langfuse (Optional)'),
      '#open' => FALSE,
      '#description' => $this->t('Enable comprehensive AI observability and analytics with Langfuse.'),
      'install_tracing' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Install Langfuse module'),
        '#default_value' => $hasTracing,
        '#description' => $this->t('Download and enable the Langfuse tracing and analytics module.'),
        '#disabled' => TRUE,
      ],
      'configure_tracing' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Configure Langfuse connection'),
        '#default_value' => $this->isLangFuseConfigured(),
        '#description' => $this->t('Set up your Langfuse project keys and connection settings.'),
        '#disabled' => TRUE,
      ],
    ];

    // Add configure links immediately after configure checkbox when module
    // is installed.
    if ($hasTracing) {
      $section['configure_links'] = [
        '#type' => 'container',
        '#weight' => 15,
        'settings_link' => [
          '#type' => 'link',
          '#title' => $this->t('⚙️ Configure Langfuse Settings'),
          '#url' => Url::fromRoute('langfuse.settings_form'),
          '#attributes' => ['class' => ['button', 'button--primary']],
        ],
        'note' => [
          '#markup' => '<p><em>' . $this->t('Note: Navigate to Langfuse configuration to set up tracing and analytics.') . '</em></p>',
        ],
      ];
    }

    // Continue with test elements.
    $section += [
      'test_container' => [
        '#type' => 'container',
        '#attributes' => ['id' => 'ai-provider-dropsolidai-tracing-test-container'],
        '#weight' => 20,
        'test_tracing' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Test Langfuse connection'),
          '#default_value' => $this->isLangFuseTested(),
          '#description' => $this->t('Verify your tracing connection is working properly.'),
          '#disabled' => TRUE,
          '#weight' => 10,
          '#states' => [
            'visible' => [
              ':input[name="manual_setup[tracing][install_tracing]"]' => ['checked' => TRUE],
              ':input[name="manual_setup[tracing][configure_tracing]"]' => ['checked' => TRUE],
            ],
          ],
        ],
        'test_button' => [
          '#type' => 'submit',
          '#value' => $this->t('🔍 Test Tracing'),
          '#ajax' => [
            'callback' => '::testTracing',
            'wrapper' => 'ai-provider-dropsolidai-tracing-test-container',
            'progress' => [
              'type' => 'throbber',
              'message' => $this->t('Testing Langfuse connection...'),
            ],
          ],
          '#limit_validation_errors' => [],
          '#submit' => ['::noopSubmitHandler'],
          '#weight' => 20,
          '#states' => [
            'visible' => [
              ':input[name="manual_setup[tracing][install_tracing]"]' => ['checked' => TRUE],
              ':input[name="manual_setup[tracing][configure_tracing]"]' => ['checked' => TRUE],
            ],
          ],
        ],
        'test_results' => [
          '#type' => 'container',
          '#attributes' => ['id' => 'ai-provider-dropsolidai-tracing-results'],
          '#markup' => '<div class="dsai-results"><em>' . $this->t('Test results will appear here.') . '</em></div>',
          '#weight' => 30,
        ],
      ],
    ];

    // Add install links when module is not installed.
    if (!$hasTracing) {
      $section['install_links'] = [
        '#type' => 'container',
        '#weight' => 50,
        'download_link' => [
          '#type' => 'link',
          '#title' => $this->t('📥 Download Langfuse Module'),
          '#url' => Url::fromUri('https://www.drupal.org/project/langfuse'),
          '#attributes' => ['target' => '_blank', 'rel' => 'noopener'],
        ],
        'instructions' => [
          '#type' => 'details',
          '#title' => $this->t('Composer & Drush Instructions'),
          '#open' => FALSE,
          '#description' => $this->t('Run these commands to install and enable the modules.'),
          'commands' => [
            '#markup' => '<div class="dsai-commands"><code>composer require drupal/langfuse</code><br><code>drush en langfuse -y</code><br><code>drush en langfuse_ai_logging -y</code></div>',
          ],
        ],
      ];
    }

    return $section;
  }

  /**
   * Builds the AI Extras section (AI Dropsolid).
   *
   * Minimal optional section that only provides an install link to the
   * `ai_dropsolid` project on Drupal.org. No configuration or testing is
   * performed here — it's purely informational and encourages installation.
   */
  private function buildAiExtraSection(): array {
    $hasAiDropsolid = $this->moduleHandler->moduleExists('ai_dropsolid');

    $section = [
      '#type' => 'details',
      '#title' => $this->t('4. AI Extras - AI Dropsolid (Optional)'),
      '#open' => FALSE,
      '#description' => $this->t('Optional features for Dropsolid AI Provider such as custom tokenizers and reranking models.'),
      'install_ai_dropsolid' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Install AI Dropsolid module'),
        '#default_value' => $hasAiDropsolid,
        '#description' => $this->t('Enable optional Dropsolid AI extras for advanced features.'),
        '#disabled' => TRUE,
        '#weight' => 10,
      ],
    ];

    // Show install link when not present.
    if (!$hasAiDropsolid) {
      $section['install_links'] = [
        '#type' => 'container',
        '#weight' => 50,
        'download_link' => [
          '#type' => 'link',
          '#title' => $this->t('📥 Download AI Dropsolid Module'),
          '#url' => Url::fromUri('https://www.drupal.org/project/ai_dropsolid'),
          '#attributes' => ['target' => '_blank', 'rel' => 'noopener'],
        ],
        'instructions' => [
          '#type' => 'details',
          '#title' => $this->t('Composer & Drush Instructions'),
          '#open' => FALSE,
          '#description' => $this->t('Run these commands to install and enable the module.'),
          'commands' => [
            '#markup' => '<div class="dsai-commands"><code>composer require drupal/ai_dropsolid</code><br><code>drush en ai_dropsolid -y</code></div>',
          ],
        ],
      ];
    }

    return $section;
  }

  /**
   * Check if LiteLLM is properly configured.
   *
   * @return bool
   *   TRUE if LiteLLM module is installed and configured with API key and host.
   */
  private function isLiteLlmConfigured(): bool {
    // Check if module exists.
    if (!$this->moduleHandler->moduleExists('ai_provider_litellm')) {
      return FALSE;
    }

    // Get LiteLLM configuration.
    $api_key_id = $this->liteLlmConfig->get('api_key');
    $host = $this->liteLlmConfig->get('host');

    // Check if required settings are configured.
    if (empty($api_key_id) || empty($host)) {
      return FALSE;
    }

    // Check if the API key exists and has a value.
    try {
      $api_key = $this->keyRepository->getKey($api_key_id);
      if (!$api_key || empty($api_key->getKeyValue())) {
        return FALSE;
      }
    }
    catch (\Exception $e) {
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Check if LiteLLM connection has been tested successfully.
   *
   * @return bool
   *   TRUE if LiteLLM has been tested successfully recently and configured.
   */
  private function isLiteLlmTested(): bool {
    // First check if LiteLLM is properly configured.
    // If not configured, we can't consider any previous test valid.
    if (!$this->isLiteLlmConfigured()) {
      return FALSE;
    }

    $config = $this->config(self::CONFIG_NAME);
    $lastTestTime = $config->get('provider.last_test_time');
    $lastTestSuccess = $config->get('provider.last_test_success');

    // If we haven't tested recently (within 1 hour), return false.
    if (!$lastTestTime || (time() - $lastTestTime) > self::CACHE_TTL) {
      return FALSE;
    }

    return (bool) $lastTestSuccess;
  }

  /**
   * Check if Postgres vector database is properly configured.
   *
   * @return bool
   *   TRUE if Postgres VDB module is installed and configured.
   */
  private function isPostgresVdbConfigured(): bool {
    // Check if module exists.
    if (!$this->moduleHandler->moduleExists('ai_vdb_provider_postgres')) {
      return FALSE;
    }

    // Get Postgres VDB configuration.
    $host = $this->postgresVdbConfig->get('host');
    $username = $this->postgresVdbConfig->get('username');
    $password = $this->postgresVdbConfig->get('password');
    $database = $this->postgresVdbConfig->get('default_database');

    // Check if required settings are configured.
    if (empty($host) || empty($username) || empty($password) || empty($database)) {
      return FALSE;
    }

    // Check if the password key exists and has a value.
    try {
      $password_key = $this->keyRepository->getKey($password);
      if (!$password_key || empty($password_key->getKeyValue())) {
        return FALSE;
      }
    }
    catch (\Exception $e) {
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Check if Postgres vector database connection has been tested successfully.
   *
   * @return bool
   *   TRUE if Postgres VDB has been tested successfully recently.
   */
  private function isPostgresVdbTested(): bool {
    // First check if Postgres VDB is properly configured.
    // If not configured, we can't consider any previous test valid.
    if (!$this->isPostgresVdbConfigured()) {
      return FALSE;
    }

    $config = $this->config(self::CONFIG_NAME);
    $lastTestTime = $config->get('vdb.last_test_time');
    $lastTestSuccess = $config->get('vdb.last_test_success');

    // If we haven't tested recently (within 1 hour), return false.
    if (!$lastTestTime || (time() - $lastTestTime) > self::CACHE_TTL) {
      return FALSE;
    }

    return (bool) $lastTestSuccess;
  }

  /**
   * Cache the result of a LiteLLM connection test.
   *
   * @param bool $success
   *   Whether the test was successful.
   */
  private function cacheTestResult(bool $success): void {
    $config = $this->configFactory()->getEditable(self::CONFIG_NAME);
    $config
      ->set('provider.last_test_time', time())
      ->set('provider.last_test_success', $success)
      ->save();
  }

  /**
   * Cache the result of a Postgres VDB connection test.
   *
   * @param bool $success
   *   Whether the test was successful.
   */
  private function cacheVdbTestResult(bool $success): void {
    $config = $this->configFactory()->getEditable(self::CONFIG_NAME);
    $config
      ->set('vdb.last_test_time', time())
      ->set('vdb.last_test_success', $success)
      ->save();
  }

  /**
   * Check if LangFuse is properly configured.
   *
   * @return bool
   *   TRUE if LangFuse module is installed and configured.
   */
  private function isLangFuseConfigured(): bool {
    // Check if module exists.
    if (!$this->moduleHandler->moduleExists('langfuse')) {
      return FALSE;
    }

    // Check if LangFuse AI Logging sub-module is installed.
    if (!$this->moduleHandler->moduleExists('langfuse_ai_logging')) {
      return FALSE;
    }

    // Get LangFuse configuration.
    $langfuse_url = $this->langFuseConfig->get('langfuse_url');
    $auth_method = $this->langFuseConfig->get('auth_method');

    // Check if required settings are configured.
    if (empty($langfuse_url) || empty($auth_method)) {
      return FALSE;
    }

    // Check authentication method specific requirements.
    switch ($auth_method) {
      case 'bearer':
        if (empty($this->langFuseConfig->get('bearer_token'))) {
          return FALSE;
        }
        break;

      case 'basic':
        $basic_auth = $this->langFuseConfig->get('basic_auth');
        if (empty($basic_auth['username']) || empty($basic_auth['password'])) {
          return FALSE;
        }
        break;

      case 'key_pair':
        $key_pair = $this->langFuseConfig->get('key_pair');
        if (empty($key_pair['public_key']) || empty($key_pair['secret_key'])) {
          return FALSE;
        }
        break;

      case 'none':
        // No authentication required.
        break;

      default:
        return FALSE;
    }

    return TRUE;
  }

  /**
   * Check if LangFuse connection has been tested successfully.
   *
   * @return bool
   *   TRUE if LangFuse has been tested successfully recently and configured.
   */
  private function isLangFuseTested(): bool {
    // First check if LangFuse is properly configured.
    // If not configured, we can't consider any previous test valid.
    if (!$this->isLangFuseConfigured()) {
      return FALSE;
    }

    $config = $this->config(self::CONFIG_NAME);
    $lastTestTime = $config->get('tracing.last_test_time');
    $lastTestSuccess = $config->get('tracing.last_test_success');

    // If we haven't tested recently (within 1 hour), return false.
    if (!$lastTestTime || (time() - $lastTestTime) > self::CACHE_TTL) {
      return FALSE;
    }

    return (bool) $lastTestSuccess;
  }

  /**
   * Cache the result of a LangFuse connection test.
   *
   * @param bool $success
   *   Whether the test was successful.
   */
  private function cacheLangFuseTestResult(bool $success): void {
    $config = $this->configFactory()->getEditable(self::CONFIG_NAME);
    $config
      ->set('tracing.last_test_time', time())
      ->set('tracing.last_test_success', $success)
      ->save();
  }

  /**
   * Creates a standardized message render array for AJAX responses.
   *
   * @param string $message
   *   The message to display.
   * @param string $type
   *   The message type (error, warning, status).
   *
   * @return array
   *   The formatted render array message.
   */
  private function buildMessage(string $message, string $type = 'error'): array {
    return [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['dsai-results', 'messages', 'messages--' . $type],
      ],
      '#weight' => 30,
      '#markup' => $message,
    ];
  }

  /**
   * Creates a standardized success message render array for AJAX responses.
   *
   * @param string $message
   *   The success message to display.
   *
   * @return array
   *   The formatted render array message.
   */
  private function buildSuccessMessage(string $message): array {
    return $this->buildMessage($message, 'status');
  }

  /**
   * Creates a standardized warning message render array for AJAX responses.
   *
   * @param string $message
   *   The warning message to display.
   *
   * @return array
   *   The formatted render array message.
   */
  private function buildWarningMessage(string $message): array {
    return $this->buildMessage($message, 'warning');
  }

  /**
   * Creates a standardized error message render array for AJAX responses.
   *
   * @param string $message
   *   The error message to display.
   *
   * @return array
   *   The formatted render array message.
   */
  private function buildErrorMessage(string $message): array {
    return $this->buildMessage($message, 'error');
  }

  /**
   * Creates a module not installed error message.
   *
   * @param string $moduleName
   *   The human-readable name of the module.
   * @param string $machineName
   *   The machine name of the module.
   *
   * @return string
   *   The formatted error message.
   */
  private function createModuleNotInstalledError(string $moduleName, string $machineName): string {
    return (string) $this->t('❌ @module module is not installed. Please install it first.', [
      '@module' => $moduleName,
    ]);
  }

  /**
   * Creates a configuration missing error message.
   *
   * @param string $serviceName
   *   The name of the service that needs configuration.
   *
   * @return string
   *   The formatted error message.
   */
  private function createConfigurationMissingError(string $serviceName): string {
    return (string) $this->t('❌ @service is not fully configured. Please configure all required settings.', [
      '@service' => $serviceName,
    ]);
  }

  /**
   * Creates a connection failed error message.
   *
   * @param string $serviceName
   *   The name of the service that failed to connect.
   * @param string $details
   *   Additional error details.
   *
   * @return string
   *   The formatted error message.
   */
  private function createConnectionFailedError(string $serviceName, string $details = ''): string {
    $message = (string) $this->t('❌ @service connection failed.', ['@service' => $serviceName]);
    if (!empty($details)) {
      $message .= ' ' . $details;
    }
    return $message;
  }

  /**
   * Creates a connection successful message.
   *
   * @param string $serviceName
   *   The name of the service that connected successfully.
   * @param string $details
   *   Additional success details.
   *
   * @return string
   *   The formatted success message.
   */
  private function createConnectionSuccessMessage(string $serviceName, string $details = ''): string {
    $message = (string) $this->t('✅ @service connection successful!', ['@service' => $serviceName]);
    if (!empty($details)) {
      $message .= ' ' . $details;
    }
    return $message;
  }

}
