<?php

namespace Drupal\ai_provider_groq\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\ai\AiProviderPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Groq API access.
 */
class GroqConfigForm extends ConfigFormBase {

  /**
   * Config settings.
   */
  const CONFIG_NAME = 'ai_provider_groq.settings';
  const CONFIG_OVERRIDES = 'ai_provider_groq.overrides';

  /**
   * The hard coded selections matching AI Settings form.
   *
   * @var array
   */
  protected $hardcodedSelections = [
    [
      'id' => 'chat_with_image_vision',
      'actual_type' => 'chat',
      'label' => 'Chat with Image Vision',
    ],
    [
      'id' => 'chat_with_complex_json',
      'actual_type' => 'chat',
      'label' => 'Chat with Complex JSON',
    ],
  ];

  /**
   * The AI Provider service.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $aiProviderManager;

  /**
   * Constructs a new GroqConfigForm object.
   */
  final public function __construct(AiProviderPluginManager $ai_provider_manager) {
    $this->aiProviderManager = $ai_provider_manager;
  }

  /**
   * {@inheritdoc}
   */
  final public static function create(ContainerInterface $container) {
    return new static(
      $container->get('ai.provider'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'groq_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::CONFIG_NAME,
      static::CONFIG_OVERRIDES,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(static::CONFIG_NAME);

    // Get AI config to find operations using Groq.
    $ai_config = $this->config('ai.settings');
    $default_providers = $ai_config->get('default_providers') ?? [];

    // Get existing operation overrides.
    $overrides_config = $this->config(static::CONFIG_OVERRIDES);
    $operation_overrides = $overrides_config->get('operation_overrides') ?? [];

    // Load the default fallback values.
    $default_reasoning_format = $config->get('reasoning_format') ?: 'hidden';
    $default_temperature = $config->get('temperature') ?: 0.6;
    $default_max_tokens = $config->get('max_tokens') ?: 1024;
    $default_json_mode = $config->get('json_mode') ?: FALSE;

    $form['api_key'] = [
      '#type' => 'key_select',
      '#title' => $this->t('Groq API Key'),
      '#description' => $this->t('The API Key. Can be found on <a href="https://console.groq.com/keys">https://console.groq.com/keys</a>.'),
      '#default_value' => $config->get('api_key'),
    ];

    // Get operation types from the AI module.
    $operation_types = $this->aiProviderManager->getOperationTypes();

    // Merge with hardcoded selections to ensure we have proper labels.
    $operation_types_map = [];
    foreach ($operation_types as $op_type) {
      $operation_types_map[$op_type['id']] = $op_type['label'];
    }

    // Add hardcoded selections to the map.
    foreach ($this->hardcodedSelections as $selection) {
      $operation_types_map[$selection['id']] = $selection['label'];
    }

    // Find operations that use Groq.
    $groq_operations = [];
    foreach ($default_providers as $operation_id => $provider_data) {
      if (isset($provider_data['provider_id']) && $provider_data['provider_id'] === 'groq') {
        $groq_operations[$operation_id] = [
          'model' => $provider_data['model_id'] ?? '',
          'label' => $operation_types_map[$operation_id] ?? $operation_id,
        ];
      }
    }

    // Common reasoning format description with improved formatting.
    $reasoning_format_description = $this->t('Controls how reasoning is formatted in responses:<ul>
      <li><strong>Parsed</strong> - Separates reasoning into a dedicated field while keeping the response concise</li>
      <li><strong>Raw</strong> - Includes reasoning within think tags in the content</li>
      <li><strong>Hidden</strong> - Returns only the final answer without any reasoning</li>
    </ul>');

    // Add Default Fallback Settings fieldset.
    $form['default_fallback_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Default Fallback Settings'),
      '#description' => $this->t('Configure default settings for Groq API requests.'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    $form['default_fallback_settings']['reasoning_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Reasoning Format'),
      '#description' => $reasoning_format_description,
      '#options' => [
        'parsed' => $this->t('Parsed'),
        'raw' => $this->t('Raw'),
        'hidden' => $this->t('Hidden'),
      ],
      '#default_value' => $default_reasoning_format,
    ];

    $form['default_fallback_settings']['temperature'] = [
      '#type' => 'number',
      '#title' => $this->t('Temperature'),
      '#description' => $this->t('Controls randomness in responses. Lower values make responses more deterministic. (Range: 0.0 - 2.0)'),
      '#default_value' => $default_temperature,
      '#min' => 0.0,
      '#max' => 2.0,
      '#step' => 0.1,
    ];

    $form['default_fallback_settings']['max_tokens'] = [
      '#type' => 'number',
      '#title' => $this->t('Max Tokens'),
      '#description' => $this->t("Maximum length of model's response. Increase for detailed step-by-step solutions."),
      '#default_value' => $default_max_tokens,
      '#min' => 1,
      '#step' => 1,
    ];

    $form['default_fallback_settings']['json_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('JSON Mode'),
      '#description' => $this->t('Enable JSON mode for structured output.'),
      '#default_value' => $default_json_mode,
    ];

    // Add per-operation settings if any operations use Groq.
    if (!empty($groq_operations)) {
      $form['operation_settings'] = [
        '#type' => 'details',
        '#title' => $this->t('Operation-Specific Settings'),
        '#description' => $this->t('Configure Groq settings for specific operations where Groq is selected as the provider.'),
        '#open' => !empty($operation_overrides),
        '#tree' => TRUE,
      ];

      foreach ($groq_operations as $operation_id => $operation_info) {
        $has_override = !empty($operation_overrides[$operation_id]);

        $form['operation_settings'][$operation_id] = [
          '#type' => 'details',
          '#title' => $operation_info['label'],
          '#description' => $this->t('Model: @model', ['@model' => $operation_info['model']]),
          '#open' => $has_override,
        ];

        // Enable overrides checkbox.
        $form['operation_settings'][$operation_id]['enable_override'] = [
          '#type' => 'checkbox',
          '#title' => $this->t('Override default settings for this operation'),
          '#default_value' => $has_override,
        ];

        // Get override values or use defaults.
        $override_values = $has_override ? $operation_overrides[$operation_id] : [];

        // Override settings (only visible when the checkbox is checked)
        $form['operation_settings'][$operation_id]['settings'] = [
          '#type' => 'container',
          '#states' => [
            'visible' => [
              ':input[name="operation_settings[' . $operation_id . '][enable_override]"]' => ['checked' => TRUE],
            ],
          ],
        ];

        // Reasoning Format setting.
        $form['operation_settings'][$operation_id]['settings']['reasoning_format'] = [
          '#type' => 'select',
          '#title' => $this->t('Reasoning Format'),
          '#description' => $reasoning_format_description,
          '#options' => [
            'parsed' => $this->t('Parsed'),
            'raw' => $this->t('Raw'),
            'hidden' => $this->t('Hidden'),
          ],
          '#default_value' => $override_values['reasoning_format'] ?? $default_reasoning_format,
        ];

        // Temperature setting.
        $form['operation_settings'][$operation_id]['settings']['temperature'] = [
          '#type' => 'number',
          '#title' => $this->t('Temperature'),
          '#description' => $this->t('Controls randomness in responses. Lower values make responses more deterministic. (Range: 0.0 - 2.0)'),
          '#default_value' => $override_values['temperature'] ?? $default_temperature,
          '#min' => 0.0,
          '#max' => 2.0,
          '#step' => 0.1,
        ];

        // Max Completion Tokens setting.
        $form['operation_settings'][$operation_id]['settings']['max_tokens'] = [
          '#type' => 'number',
          '#title' => $this->t('Max Tokens'),
          '#description' => $this->t("Maximum length of model's response. Increase for detailed step-by-step solutions."),
          '#default_value' => $override_values['max_tokens'] ?? $default_max_tokens,
          '#min' => 1,
          '#step' => 1,
        ];

        // JSON Mode setting.
        $form['operation_settings'][$operation_id]['settings']['json_mode'] = [
          '#type' => 'checkbox',
          '#title' => $this->t('JSON Mode'),
          '#description' => $this->t('Enable JSON mode for structured output.'),
          '#default_value' => $override_values['json_mode'] ?? $default_json_mode,
        ];
      }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Save main settings.
    $this->config(static::CONFIG_NAME)
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('reasoning_format', $form_state->getValue(['default_fallback_settings', 'reasoning_format']))
      ->set('temperature', $form_state->getValue(['default_fallback_settings', 'temperature']))
      ->set('max_tokens', $form_state->getValue(['default_fallback_settings', 'max_tokens']))
      ->set('json_mode', $form_state->getValue(['default_fallback_settings', 'json_mode']))
      ->save();

    // Process and save operation-specific overrides.
    $operation_settings = $form_state->getValue('operation_settings', []);
    $overrides = [];

    foreach ($operation_settings as $operation_id => $settings) {
      if (!empty($settings['enable_override'])) {
        $overrides[$operation_id] = [
          'reasoning_format' => $settings['settings']['reasoning_format'],
          'temperature' => $settings['settings']['temperature'],
          'max_tokens' => $settings['settings']['max_tokens'],
          'json_mode' => $settings['settings']['json_mode'],
        ];
      }
    }

    $this->config(static::CONFIG_OVERRIDES)
      ->set('operation_overrides', $overrides)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
