<?php

namespace Drupal\ai_provider_groq\Plugin\AiProvider;

use Drupal\ai\Base\OpenAiBasedProviderClientBase;
use Drupal\Component\Serialization\Json;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\AiProvider;
use Drupal\ai\Enum\AiModelCapability;
use Drupal\ai\Traits\OperationType\ChatTrait;
use Drupal\Component\Utility\Crypt;

/**
 * Plugin implementation of the 'groq' provider.
 */
#[AiProvider(
  id: 'groq',
  label: new TranslatableMarkup('Groq'),
)]
class GroqProvider extends OpenAiBasedProviderClientBase {

  use ChatTrait;

  /**
   * {@inheritdoc}
   */
  protected string $endpoint = 'https://api.groq.com/openai/v1';

  /**
   * The list of models that support function calling.
   *
   * @var array
   */
  protected array $toolCallingModels = [
    'qwen-qwq-32b',
    'qwen-2.5-coder-32b',
    'qwen-2.5-32b',
    'deepseek-r1-distill-qwen-32b',
    'deepseek-r1-distill-llama-70b',
    'llama-3.3-70b-versatile',
    'llama-3.1-8b-instant',
    'mixtral-8x7b-32768',
    'gemma2-9b-it',
  ];

  /**
   * {@inheritdoc}
   */
  public function getConfiguredModels(?string $operation_type = NULL, array $capabilities = []): array {
    // Get the cache key.
    $cache_key = 'groq_models_' . $operation_type . '_' . Crypt::hashBase64(Json::encode($capabilities));
    $cache_data = $this->cacheBackend->get($cache_key);

    if (!empty($cache_data)) {
      return $cache_data->data;
    }

    // Get all available models with a single request.
    $response = $this->getClient()->models()->list()->toArray();

    $models = [];

    if (isset($response['data'])) {
      foreach ($response['data'] as $model) {
        $model_id = $model['id'];

        // Skip text-to-speech models (for now).
        if (strpos($model_id, 'playai-tts') === 0) {
          continue;
        }

        // Skip speech-to-text (Whisper) models.
        if (strpos($model_id, 'whisper') === 0 || strpos($model_id, 'distil-whisper') === 0) {
          continue;
        }

        // Handle vision models.
        if (in_array(AiModelCapability::ChatWithImageVision, $capabilities, TRUE)) {
          // Only include models with "vision" in their name.
          if (strpos($model_id, 'vision') !== FALSE) {
            $models[$model_id] = $model_id;
          }
          continue;
        }

        // Add all other models.
        $models[$model_id] = $model_id;
      }
    }

    // If its function calling capability.
    if (in_array(AiModelCapability::ChatTools, $capabilities, TRUE)) {
      // Hardcoded list :(.
      foreach ($models as $model_id => $model_name) {
        if (!in_array($model_id, $this->toolCallingModels)) {
          unset($models[$model_id]);
        }
      }
    }

    // Save cache.
    if (!empty($models)) {
      asort($models);
      $this->cacheBackend->set($cache_key, $models);
    }

    return $models;
  }

  /**
   * Determines if a model supports reasoning format.
   *
   * @param string $model_id
   *   The model ID to check.
   *
   * @return bool
   *   TRUE if the model supports reasoning format, FALSE otherwise.
   */
  protected function modelSupportsReasoning(string $model_id): bool {
    // Check if the model ID contains '-qwq-' or 'deepseek'.
    return strpos($model_id, '-qwq-') !== FALSE || strpos($model_id, 'deepseek') !== FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedOperationTypes(): array {
    return [
      'chat',
    ];
  }

  /**
   * Get operation-specific settings or default settings.
   *
   * @param string $operation_type
   *   The type of operation.
   *
   * @return array
   *   Array of settings for the operation.
   */
  protected function getOperationSettings(string $operation_type): array {
    // Get default fallback settings.
    $config = $this->getConfig();
    $default_settings = [
      'reasoning_format' => $config->get('reasoning_format') ?: 'hidden',
      'temperature' => $config->get('temperature') ?: 0.6,
      'max_tokens' => $config->get('max_tokens') ?: 1024,
      'json_mode' => $config->get('json_mode') ?: FALSE,
    ];

    // Get operation-specific overrides if they exist.
    $overrides_config = $this->configFactory->get('ai_provider_groq.overrides');
    $operation_overrides = $overrides_config->get('operation_overrides') ?? [];

    // If there are overrides for this operation, merge them with defaults.
    if (!empty($operation_overrides[$operation_type])) {
      return array_merge($default_settings, $operation_overrides[$operation_type]);
    }

    return $default_settings;
  }

  /**
   * {@inheritdoc}
   */
  public function getModelSettings(string $model_id, array $generalConfig = []): array {
    // If this is from a specific operation, get operation-specific settings.
    $operation_type = $generalConfig['operation_type'] ?? NULL;
    if ($operation_type) {
      $settings = $this->getOperationSettings($operation_type);
      return array_merge($generalConfig, $settings);
    }
    return $generalConfig;
  }

  /**
   * {@inheritdoc}
   */
  public function getSetupData(): array {
    return [
      'key_config_name' => 'api_key',
      'default_models' => [
        'chat' => 'llama-3.3-70b-versatile',
      ],
    ];
  }

}
