<?php

namespace Drupal\ai_provider_infomaniak\Form;

use Drupal\ai\AiProviderPluginManager;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Infomaniak AI API access.
 */
class InfomaniakConfigForm extends ConfigFormBase {

  /**
   * Config settings.
   */
  public const CONFIG_NAME = 'ai_provider_infomaniak.settings';

  /**
   * The AI Provider manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected AiProviderPluginManager $providerManager;

  /**
   * Constructs a new InfomaniakConfigForm object.
   */
  final public function __construct(AiProviderPluginManager $provider_manager) {
    $this->providerManager = $provider_manager;
  }

  /**
   * {@inheritdoc}
   */
  final public static function create(ContainerInterface $container) {
    return new static(
      $container->get('ai.provider'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'infomaniak_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::CONFIG_NAME,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(static::CONFIG_NAME);

    // Getting Started section.
    $form['getting_started'] = [
      '#type' => 'details',
      '#title' => $this->t('Getting Started'),
      '#open' => !$config->get('api_key') || !$config->get('product_id'),
      '#weight' => -10,
    ];

    $form['getting_started']['info'] = [
      '#type' => 'markup',
      '#markup' => $this->t('
        <div class="messages messages--status">
          <h3>Welcome to Infomaniak AI Provider Setup</h3>
          <p>This module integrates with Infomaniak\'s AI API services.</p>

          <h4>📋 What You Need</h4>
          <ol>
            <li><strong>Infomaniak Account</strong> with AI Tools enabled</li>
            <li><strong>API Token</strong> for authentication</li>
            <li><strong>Product ID</strong> from your AI Tools dashboard</li>
          </ol>

          <h4>🔑 Step 1: Create Your API Token</h4>
          <ol>
            <li>Log in to <a href="@infomaniak_manager" target="_blank">Infomaniak Manager</a></li>
            <li>Click on your profile (top right) → <strong>My Profile</strong></li>
            <li>Go to <strong>API tokens</strong></li>
            <li>Click <strong>Create a token</strong></li>
            <li>Configure the token:
              <ul>
                <li><strong>Token name:</strong> Choose a name (e.g., "Drupal AI")</li>
                <li><strong>Application:</strong> Select your application or "Default application"</li>
                <li><strong>Scopes:</strong> Select <strong>"ai-tools - AI Tools"</strong></li>
                <li><strong>Valid for:</strong> Choose expiration (or "Unlimited")</li>
              </ul>
            </li>
            <li>Click <strong>Create</strong> and copy the generated token</li>
            <li>Save the token in Drupal\'s Key module (recommended) or note it securely</li>
          </ol>

          <h4>🆔 Step 2: Get Your Product ID</h4>
          <ol>
            <li>In <a href="@infomaniak_manager" target="_blank">Infomaniak Manager</a>, go to <strong>Cloud Computing</strong></li>
            <li>Click on the <strong>AI Tools</strong> tab</li>
            <li>In the left sidebar under "AI Tools", you will see the <strong>Product ID</strong> field</li>
            <li>Copy the Product ID (it\'s a numeric value like "123456")</li>
          </ol>

          <h4>✅ Step 3: Configure Below</h4>
          <ol>
            <li>Enter your <strong>API Token</strong> in the field below</li>
            <li>Enter your <strong>Product ID</strong></li>
            <li>Click <strong>"Test Connection"</strong> to verify your credentials</li>
            <li>Once connected, the available models will be displayed automatically</li>
            <li>Configure model settings if needed</li>
            <li>Save the form</li>
          </ol>

          <h4>🆘 Need Help?</h4>
          <ul>
            <li><a href="https://www.infomaniak.com/en/support" target="_blank">Infomaniak Support Center</a></li>
            <li><a href="https://faq.infomaniak.com/" target="_blank">Infomaniak FAQ</a></li>
          </ul>
        </div>
      ', [
        '@infomaniak_manager' => 'https://manager.infomaniak.com/',
      ]),
    ];

    // Connection Status.
    $form['connection_status'] = [
      '#type' => 'details',
      '#title' => $this->t('Connection Status'),
      '#open' => TRUE,
      '#weight' => -5,
    ];

    $status = $this->getConnectionStatus($config);
    $form['connection_status']['status'] = [
      '#type' => 'markup',
      '#markup' => $status['markup'],
    ];

    // Determine if connection is ready.
    $is_ready = $status['status'] === 'ready';

    // API Credentials section - Hide when connection is ready.
    $form['credentials'] = [
      '#type' => 'details',
      '#title' => $this->t('API Credentials'),
      '#open' => !$is_ready,
      '#description' => $this->t('Your Infomaniak AI API authentication details.'),
    ];

    $form['credentials']['api_key'] = [
      '#type' => 'key_select',
      '#title' => $this->t('API Key'),
      '#description' => $this->t('
        <br><strong>Where to find your token (key):</strong> See "Getting started" section.
      '),
      '#default_value' => $config->get('api_key'),
      '#required' => TRUE,
    ];

    $form['credentials']['product_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Product ID'),
      '#description' => $this->t('
        <br><strong>Where to find it:</strong> See "Getting started" section.
      '),
      '#default_value' => $config->get('product_id'),
      '#required' => TRUE,
      '#placeholder' => '123456',
      '#attributes' => [
        'autocomplete' => 'off',
      ],
    ];

    $form['credentials']['test_connection'] = [
      '#type' => 'button',
      '#value' => $this->t('Test Connection'),
      '#ajax' => [
        'callback' => '::testConnection',
        'wrapper' => 'connection-test-result',
        'effect' => 'fade',
      ],
      '#limit_validation_errors' => [],
      '#attributes' => [
        'class' => ['button--primary'],
      ],
    ];

    $form['credentials']['connection_result'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'connection-test-result'],
    ];

    // Models configuration section.
    $form['models'] = [
      '#type' => 'details',
      '#title' => $this->t('Models Settings'),
      '#open' => $is_ready,
      '#description' => $this->t('Configure available AI models from Infomaniak.'),
    ];

    // Load models from configuration.
    $models_config = $this->configFactory()->get('ai_provider_infomaniak.models');
    $definitions = $models_config->get('definitions') ?? [];

    // Map to simpler structure for table display.
    $all_models_data = array_map(
      fn(array $def) => [
        'id' => $def['id'],
        'name' => $def['name'],
        'type' => $def['type'],
      ],
      $definitions
    );

    $form['models']['table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Model ID'),
        $this->t('Name'),
        $this->t('Type'),
        $this->t('Actions'),
      ],
      '#empty' => $this->t('No models available.'),
    ];

    foreach ($all_models_data as $index => $model) {
      $form['models']['table'][$index] = [
        'id' => [
          '#markup' => '<code>' . $model['id'] . '</code>',
        ],
        'name' => [
          '#markup' => $model['name'],
        ],
        'type' => [
          '#markup' => $model['type'],
        ],
        'actions' => [
          '#type' => 'operations',
          '#links' => [
            'edit' => [
              'title' => $this->t('Edit'),
              'url' => Url::fromRoute('ai_provider_infomaniak.edit_model', [], [
                'query' => ['model_id' => $model['id']],
              ]),
            ],
          ],
        ],
      ];
    }

    // Advanced Settings section.
    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced Settings'),
      '#open' => FALSE,
      '#weight' => 100,
    ];

    $form['advanced']['base_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Base URL'),
      '#description' => $this->t('
        The base URL for Infomaniak AI API.<br>
        <strong>Default:</strong> <code>https://api.infomaniak.com/1/ai</code><br>
        <strong>Warning:</strong> Only change this if you are using a different Infomaniak API endpoint or custom proxy.
      '),
      '#default_value' => $config->get('base_url') ?: 'https://api.infomaniak.com/1/ai',
      '#placeholder' => 'https://api.infomaniak.com/1/ai',
    ];

    $form['advanced']['organization'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Organization (Optional)'),
      '#description' => $this->t('Optional organization identifier if required by your Infomaniak account setup.'),
      '#default_value' => $config->get('organization'),
      '#placeholder' => $this->t('Leave empty if not applicable'),
    ];

    $form['advanced']['timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Request Timeout'),
      '#description' => $this->t('Maximum time in seconds to wait for API responses. Recommended: 60-120 seconds for image generation, 30-60 for chat.'),
      '#default_value' => $config->get('timeout') ?: 60,
      '#min' => 1,
      '#max' => 300,
      '#field_suffix' => $this->t('seconds'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Ajax callback to test the connection.
   */
  public function testConnection(array &$form, FormStateInterface $form_state) {
    $api_key = $form_state->getValue('api_key');
    $product_id = $form_state->getValue('product_id');
    $base_url = $form_state->getValue('base_url') ?: 'https://api.infomaniak.com/1/ai';

    if (empty($api_key) || empty($product_id)) {
      $form['credentials']['connection_result']['message'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--error" role="alert">
          <strong>Error:</strong> Please provide both API Key and Product ID before testing the connection.
        </div>',
      ];
      return $form['credentials']['connection_result'];
    }

    try {
      $provider = $this->providerManager->createInstance('infomaniak');
      $provider->setAuthentication([
        'api_key' => $this->loadApiKeyValue($api_key),
        'product_id' => $product_id,
      ]);

      $endpoint = sprintf('%s/%s/openai', rtrim($base_url, '/'), $product_id);

      $form['credentials']['connection_result']['message'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--status" role="status">
          <h4>✅ Connection Successful!</h4>
          <ul>
            <li><strong>Endpoint:</strong> <code>' . htmlspecialchars($endpoint) . '</code></li>
            <li><strong>Product ID:</strong> <code>' . htmlspecialchars($product_id) . '</code></li>
            <li><strong>Status:</strong> API credentials are valid and the provider is configured correctly.</li>
          </ul>
          <p><em>You can now click "Discover Models" to automatically detect available models!</em></p>
        </div>',
      ];
    }
    catch (\Exception $e) {
      $form['credentials']['connection_result']['message'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--error" role="alert">
          <h4>❌ Connection Failed</h4>
          <p><strong>Error:</strong> ' . htmlspecialchars($e->getMessage()) . '</p>
          <p><strong>Troubleshooting:</strong></p>
          <ul>
            <li>Verify your API key is correct and active</li>
            <li>Confirm your Product ID matches your Infomaniak AI product</li>
            <li>Check that your Infomaniak AI subscription is active</li>
            <li>Ensure the API endpoint is accessible from your server</li>
          </ul>
        </div>',
      ];
    }

    return $form['credentials']['connection_result'];
  }

  /**
   * Load the actual API key value from the key module.
   *
   * @param string $keyId
   *   The key ID to load.
   *
   * @return string
   *   The API key value.
   */
  protected function loadApiKeyValue(string $keyId): string {
    if (empty($keyId)) {
      return '';
    }

    $keyRepository = \Drupal::service('key.repository');
    $key = $keyRepository->getKey($keyId);

    if ($key) {
      return $key->getKeyValue();
    }

    return '';
  }

  /**
   * Get connection status information.
   */
  protected function getConnectionStatus($config) {
    $api_key = $config->get('api_key');
    $product_id = $config->get('product_id');

    if (!$api_key || !$product_id) {
      return [
        'status' => 'not_configured',
        'markup' => '<div class="messages messages--warning">
          <strong>⚠️ Not Configured</strong><br>
          Please configure your API credentials below to enable this provider.
        </div>',
      ];
    }

    try {
      $provider = $this->providerManager->createInstance('infomaniak');
      if ($provider->isUsable()) {
        $endpoint = sprintf(
          '%s/%s/openai',
          rtrim($config->get('base_url') ?: 'https://api.infomaniak.com/1/ai', '/'),
          $product_id
        );

        return [
          'status' => 'ready',
          'markup' => '<div class="messages messages--status">
            <strong>✅ Ready to Use</strong><br>
            API Endpoint: <code>' . htmlspecialchars($endpoint) . '</code><br>
            The provider is configured and ready for use.
          </div>',
        ];
      }
    }
    catch (\Exception $e) {
      // Ignore errors in status check.
    }

    return [
      'status' => 'configured',
      'markup' => '<div class="messages messages--warning">
        <strong>⚙️ Configured</strong><br>
        Credentials are set. Click "Test Connection" to verify they work correctly.
      </div>',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Save configuration.
    $this->config(static::CONFIG_NAME)
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('product_id', $form_state->getValue('product_id'))
      ->set('base_url', $form_state->getValue('base_url'))
      ->set('organization', $form_state->getValue('organization'))
      ->set('timeout', $form_state->getValue('timeout'))
      ->save();

    // Clear model cache.
    $this->clearModelCache();

    // Set default models.
    $this->setDefaultModels();

    parent::submitForm($form, $form_state);
  }

  /**
   * Clears the model cache for this provider.
   */
  protected function clearModelCache(): void {
    $cache_backend = \Drupal::cache();

    $cache_backend->delete('ai_provider_infomaniak:models:all');
    $cache_backend->delete('ai_provider_infomaniak:models:chat');
    $cache_backend->delete('ai_provider_infomaniak:models:text_to_image');
    $cache_backend->delete('ai_provider_infomaniak:models:speech_to_text');
    $cache_backend->delete('ai_provider_infomaniak:models:text_to_speech');
    $cache_backend->delete('ai_provider_infomaniak:models:image_to_image');
    $cache_backend->delete('ai_provider_infomaniak:models:embeddings');
    $cache_backend->delete('ai_provider_infomaniak:models:moderation');
  }

  /**
   * Sets default models using the provider's setup data.
   */
  protected function setDefaultModels(): void {
    try {
      $provider = $this->providerManager->createInstance('infomaniak');

      if (is_callable([$provider, 'getSetupData'])) {
        $setupData = $provider->getSetupData();

        if (!empty($setupData['default_models'])) {
          foreach ($setupData['default_models'] as $operationType => $modelId) {
            $this->providerManager->defaultIfNone($operationType, 'infomaniak', $modelId);
          }
        }
      }
    }
    catch (\Exception $e) {
      \Drupal::logger('ai_provider_infomaniak')->warning(
        'Failed to set default models: @error',
        ['@error' => $e->getMessage()]
      );
    }
  }

}
