# Infomaniak AI Provider for Drupal

**Project page:** https://www.drupal.org/project/ai_provider_infomaniak

Drupal AI module provider for Infomaniak's AI services, providing access to state-of-the-art open-source AI models hosted in Switzerland.

## Overview

This module integrates Infomaniak's AI Tools with Drupal's AI module, enabling you to use various AI models for:
- **Chat/Text Generation** - Conversational AI and text completion
- **Image Generation** - Text-to-image creation
- **Image Editing** - AI-powered image-to-image transformations
- **Embeddings** - Text embedding for semantic search

## Requirements

- Drupal 10.3+ or Drupal 11+
- PHP 8.1+
- [Drupal AI module](https://www.drupal.org/project/ai) (drupal/ai) version 1.2.4+
- Infomaniak account with AI Tools enabled

## Installation

1. **Install via Composer:**
   ```bash
   composer require drupal/ai_provider_infomaniak
   ```

2. **Enable the modules:**
   ```bash
   drush en ai ai_provider_infomaniak -y
   drush cr
   ```

   The AI module will be installed automatically as a dependency.

## Configuration

### Step 1: Get Your Infomaniak Credentials

#### Create an API Token

1. Log in to [Infomaniak Manager](https://manager.infomaniak.com/)
2. Click on your profile (top right) → **My Profile**
3. Go to **API tokens**
4. Click **Create a token**
5. Configure:
   - **Token name:** "Drupal AI" (or your choice)
   - **Application:** Select your application or "Default application"
   - **Scopes:** Select **"ai-tools - AI Tools"**
   - **Valid for:** Choose expiration or "Unlimited"
6. Click **Create** and copy the generated token

#### Get Your Product ID

1. In Infomaniak Manager, go to **Cloud Computing**
2. Click on the **AI Tools** tab
3. In the left sidebar, find the **Product ID** field
4. Copy the Product ID (numeric value like "123456")

### Step 2: Configure in Drupal

1. Go to **Configuration** → **AI** → **Providers** → **Infomaniak**
   (`/admin/config/ai/providers/infomaniak`)

2. Enter your credentials:
   - **API Token:** Paste your token
   - **Product ID:** Enter your Product ID

3. Click **Test Connection** to verify

4. Click **Save configuration**

## Available Models

The module provides access to **Infomaniak AI models**:

### ✅ Chat Models
- **Llama 3.3** (`llama3`) - Meta's latest open-source model
- **Granite 3.1 8B** (`granite`) - IBM's enterprise AI model
- **Mistral Small 3.2 24B** (`mistral3`) - Efficient instruction-following
- **Qwen3 VL 235B** (`qwen3`) - Vision-language multimodal model
- **Gemma 3n E4B** (`gemma3n`) - Google's compact model
- **Apertus 70B Instruct** (`swiss-ai/Apertus-70B-Instruct-2509`) - Swiss AI high-performance model
- **GPT-OSS 120B** (`openai/gpt-oss-120b`) - Large open-source model

### ✅ Embedding Models
- **BGE Multilingual Gemma2** (`bge_multilingual_gemma2`) - 3584-dim multilingual embeddings
- **All MiniLM L12 v2** (`mini_lm_l12_v2`) - 384-dim fast embeddings
- **Qwen3 Embedding 8B** (`Qwen/Qwen3-Embedding-8B`) - 4096-dim high-quality vectors

### ✅ Image Generation
- **Flux Schnell** (`flux`) - Fast, high-quality image generation

## Usage Examples

### Chat

```php
$provider_manager = \Drupal::service('ai.provider');
$provider = $provider_manager->createInstance('infomaniak');

// Simple chat
$input = new ChatInput([
  new ChatMessage('user', 'Write a haiku about AI'),
]);
$response = $provider->chat($input, 'Llama 3.3');
echo $response->getNormalized();
```

### Image Generation

```php
$input = new TextToImageInput('A Swiss mountain landscape at sunset');
$response = $provider->textToImage($input, 'flux');
$images = $response->getNormalized();
file_save_data($images[0]->getBinary(), 'public://ai-generated.png');
```

### Embeddings

```php
$input = new EmbeddingsInput('Your text to embed');
$response = $provider->embeddings($input, 'mini_lm_l12_v2');
$vector = $response->getNormalized();
// $vector is a flat array of floats (e.g., 384 dimensions)
```

## Model Configuration

Each model can be configured individually:

1. Go to **Configuration** → **AI** → **Providers** → **Infomaniak**
2. In the **Models Settings** section, click **Edit** next to any model
3. Configure:
   - **Temperature** (0.0 - 2.0) - Controls randomness
   - **Max Tokens** - Maximum response length
   - **Top P** (0.0 - 1.0) - Nucleus sampling
   - **Frequency Penalty** (-2.0 - 2.0) - Reduce repetition
   - **Presence Penalty** (-2.0 - 2.0) - Encourage topic diversity

## Unsupported Models

The following models are not currently available:

- **Reranking Models (2)** - BGE Reranker v2 M3, Qwen3 Reranker 0.6B
- **Speech-to-Text** - Whisper V3
- **Image Generation** - Photomaker V2

📖 **For detailed technical information and implementation possibilities**, see [UNSUPPORTED_MODELS.md](UNSUPPORTED_MODELS.md)

## Permissions

- **Administer AI Providers** - Required to configure the module
- **Autocomplete Infomaniak Model List** - For model selection interfaces

## Support & Contributing

### Getting Help

- **Issue Queue:** https://www.drupal.org/project/issues/ai_provider_infomaniak
- **Infomaniak Support:** https://www.infomaniak.com/en/support
- **Infomaniak FAQ:** https://faq.infomaniak.com/
- **Drupal AI Module:** https://www.drupal.org/project/ai
- **AI Module Documentation:** https://project.pages.drupalcode.org/ai/

### Contributing

Contributions are welcome! Please:
1. Submit issues and feature requests in the [issue queue](https://www.drupal.org/project/issues/ai_provider_infomaniak)
2. Submit patches following [Drupal coding standards](https://www.drupal.org/docs/develop/standards)
3. Support the AI module ecosystem via [OpenCollective](https://opencollective.com/drupal-ai)

## Credits

- **Infomaniak** - AI Services provider (https://www.infomaniak.com)
- **Drupal AI Module** - Base AI integration framework (https://www.drupal.org/project/ai)
- Module developed for the Drupal community
