<?php

namespace Drupal\ai_provider_infomaniak\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Form for editing model settings.
 *
 * Uses query parameters instead of path parameters to handle model IDs
 * that contain special characters (like slashes).
 */
class ModelEditForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['ai_provider_infomaniak.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_provider_infomaniak_model_edit_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, Request $request = NULL) {
    $request = $request ?? \Drupal::request();
    $model_id = $request->query->get('model_id');

    if (empty($model_id)) {
      $this->messenger()->addError($this->t('No model ID specified.'));
      return $form;
    }

    // Store model ID in form state for submit handler.
    $form_state->set('model_id', $model_id);

    // Load model name from configuration.
    $models_config = $this->configFactory()->get('ai_provider_infomaniak.models');
    $definitions = $models_config->get('definitions') ?? [];
    $models = array_column($definitions, 'name', 'id');
    $model_name = $models[$model_id] ?? $model_id;

    $form['model_info'] = [
      '#type' => 'item',
      '#title' => $this->t('Editing model'),
      '#markup' => '<strong>' . $model_name . '</strong> (<code>' .
        $model_id . '</code>)',
    ];

    // Get current configuration.
    $config = $this->config('ai_provider_infomaniak.settings');
    $model_config_key = 'models.' . str_replace('/', '_', $model_id);
    $current_config = $config->get($model_config_key) ?? [];

    // Configuration fields.
    $form['configuration'] = [
      '#type' => 'details',
      '#title' => $this->t('Model Configuration'),
      '#open' => TRUE,
    ];

    $form['configuration']['temperature'] = [
      '#type' => 'number',
      '#title' => $this->t('Temperature'),
      '#description' => $this->t('Controls randomness. Lower values make output more focused and deterministic. Range: 0.0 to 2.0.'),
      '#default_value' => $current_config['temperature'] ?? 0.7,
      '#min' => 0,
      '#max' => 2,
      '#step' => 0.1,
    ];

    $form['configuration']['max_tokens'] = [
      '#type' => 'number',
      '#title' => $this->t('Max Tokens'),
      '#description' => $this->t('Maximum number of tokens to generate in the response.'),
      '#default_value' => $current_config['max_tokens'] ?? 2048,
      '#min' => 1,
      '#max' => 32768,
    ];

    $form['configuration']['top_p'] = [
      '#type' => 'number',
      '#title' => $this->t('Top P'),
      '#description' => $this->t('Nucleus sampling parameter. Range: 0.0 to 1.0'),
      '#default_value' => $current_config['top_p'] ?? 1.0,
      '#min' => 0,
      '#max' => 1,
      '#step' => 0.01,
    ];

    $form['configuration']['frequency_penalty'] = [
      '#type' => 'number',
      '#title' => $this->t('Frequency Penalty'),
      '#description' => $this->t('Reduces repetition of tokens based on their frequency. Range: -2.0 to 2.0'),
      '#default_value' => $current_config['frequency_penalty'] ?? 0,
      '#min' => -2,
      '#max' => 2,
      '#step' => 0.1,
    ];

    $form['configuration']['presence_penalty'] = [
      '#type' => 'number',
      '#title' => $this->t('Presence Penalty'),
      '#description' => $this->t('Reduces repetition of topics. Range: -2.0 to 2.0'),
      '#default_value' => $current_config['presence_penalty'] ?? 0,
      '#min' => -2,
      '#max' => 2,
      '#step' => 0.1,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $model_id = $form_state->get('model_id');
    $config = $this->config('ai_provider_infomaniak.settings');

    // Use underscore instead of slash for config key.
    $model_config_key = 'models.' . str_replace('/', '_', $model_id);

    // Save the configuration.
    $config->set($model_config_key, [
      'temperature' => (float) $form_state->getValue('temperature'),
      'max_tokens' => (int) $form_state->getValue('max_tokens'),
      'top_p' => (float) $form_state->getValue('top_p'),
      'frequency_penalty' => (float) $form_state->getValue('frequency_penalty'),
      'presence_penalty' => (float) $form_state->getValue('presence_penalty'),
    ])->save();

    parent::submitForm($form, $form_state);

    // Redirect back to the main config form.
    $form_state->setRedirect('ai_provider_infomaniak.settings_form');
  }

}
