<?php

namespace Drupal\ai_provider_infomaniak\Plugin\AiProvider;

use Drupal\ai\Attribute\AiProvider;
use Drupal\ai\Base\OpenAiBasedProviderClientBase;
use Drupal\ai\Enum\AiProviderCapability;
use Drupal\ai\Exception\AiQuotaException;
use Drupal\ai\Exception\AiRateLimitException;
use Drupal\ai\Exception\AiResponseErrorException;
use Drupal\ai\Exception\AiSetupFailureException;
use Drupal\ai\OperationType\GenericType\ImageFile;
use Drupal\ai\OperationType\ImageToImage\ImageToImageInput;
use Drupal\ai\OperationType\ImageToImage\ImageToImageInterface;
use Drupal\ai\OperationType\ImageToImage\ImageToImageOutput;
use Drupal\ai\OperationType\TextToImage\TextToImageInput;
use Drupal\ai\OperationType\TextToImage\TextToImageOutput;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the 'infomaniak' provider.
 */
#[AiProvider(
  id: 'infomaniak',
  label: new TranslatableMarkup('Infomaniak AI'),
)]
class InfomaniakProvider extends OpenAiBasedProviderClientBase implements
  ContainerFactoryPluginInterface,
  ImageToImageInterface {

  /**
   * Product ID for Infomaniak API.
   *
   * @var string
   */
  protected string $productId = '';

  /**
   * Infomaniak provides predefined OpenAI-compatible models.
   *
   * @var bool
   */
  protected bool $hasPredefinedModels = TRUE;

  /**
   * Get all model definitions from configuration.
   *
   * @return array
   *   Array of model definitions.
   */
  protected function getModelDefinitions(): array {
    return $this->configFactory->get('ai_provider_infomaniak.models')->get('definitions') ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguredModels(?string $operation_type = NULL, array $capabilities = []): array {
    $definitions = $this->getModelDefinitions();

    // Filter by operation type if specified.
    if ($operation_type) {
      $definitions = array_filter(
        $definitions,
        fn(array $def) => $def['operation_type'] === $operation_type
      );
    }

    // Return as id => name array.
    return array_column($definitions, 'name', 'id');
  }

  /**
   * {@inheritdoc}
   */
  public function isUsable(?string $operation_type = NULL, array $capabilities = []): bool {
    // Check if API key is configured.
    if (!$this->hasAuthentication() && !$this->getConfig()->get('api_key')) {
      return FALSE;
    }

    // Check if product ID is configured.
    if (!$this->getConfig()->get('product_id')) {
      return FALSE;
    }

    // If operation type is specified, check if it's supported.
    if ($operation_type) {
      return in_array($operation_type, $this->getSupportedOperationTypes(), TRUE);
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedOperationTypes(): array {
    return [
      'chat',
      'text_to_image',
      'speech_to_text',
      'text_to_speech',
      'image_to_image',
      'embeddings',
      'moderation',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedCapabilities(): array {
    return [
      AiProviderCapability::StreamChatOutput,
      AiProviderCapability::ChatFiberSupport,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig(): ImmutableConfig {
    return $this->configFactory->get('ai_provider_infomaniak.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function setAuthentication(mixed $authentication): void {
    // Handle both string and array authentication.
    if (is_array($authentication)) {
      $this->apiKey = $authentication['api_key'] ?? '';
      $this->productId = $authentication['product_id'] ?? '';
    }
    else {
      $this->apiKey = $authentication;
    }
    $this->client = NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getModelSettings(string $model_id, array $generalConfig = []): array {
    return $generalConfig;
  }

  /**
   * {@inheritdoc}
   */
  protected function loadClient(): void {
    if (empty($this->client)) {
      if (!$this->hasAuthentication()) {
        try {
          $this->loadAuthentication();
        }
        catch (\Exception $e) {
          throw new AiSetupFailureException('Failed to authenticate with Infomaniak AI: ' . $e->getMessage(), $e->getCode(), $e);
        }
      }

      // Set the custom endpoint with product_id.
      $this->setEndpoint($this->buildEndpointUrl());

      // Create the client using parent method.
      try {
        $this->client = $this->createClient();
      }
      catch (\Exception $e) {
        throw new AiSetupFailureException('Failed to initialize Infomaniak AI client: ' . $e->getMessage(), $e->getCode(), $e);
      }
    }
  }

  /**
   * Loads authentication credentials from configuration.
   */
  protected function loadAuthentication(): void {
    $config = $this->getConfig();

    // Load API key.
    try {
      $this->apiKey = $this->loadApiKey();
    }
    catch (\Exception $e) {
      throw new AiSetupFailureException('Failed to load API key: ' . $e->getMessage(), $e->getCode(), $e);
    }

    // Load product ID.
    $this->productId = $config->get('product_id') ?? '';

    if (empty($this->productId)) {
      throw new AiSetupFailureException('Product ID is not configured for Infomaniak AI provider.');
    }
  }

  /**
   * Builds the endpoint URL with product ID.
   *
   * @return string
   *   The complete endpoint URL.
   */
  protected function buildEndpointUrl(): string {
    $baseUrl = $this->getConfig()->get('base_url') ?? 'https://api.infomaniak.com/2/ai';
    return sprintf('%s/%s/openai/v1', rtrim($baseUrl, '/'), $this->productId);
  }

  /**
   * {@inheritdoc}
   */
  public function getMaxInputTokens(string $model_id): int {
    $config = $this->getConfig();
    $models = $config->get('models') ?? [];

    foreach ($models as $model) {
      if (($model['model_id'] ?? '') === $model_id) {
        return $model['max_input_tokens'] ?? 8192;
      }
    }

    return 8192;
  }

  /**
   * {@inheritdoc}
   */
  public function getMaxOutputTokens(string $model_id): int {
    $config = $this->getConfig();
    $models = $config->get('models') ?? [];

    foreach ($models as $model) {
      if (($model['model_id'] ?? '') === $model_id) {
        return $model['max_output_tokens'] ?? 4096;
      }
    }

    return 4096;
  }

  /**
   * {@inheritdoc}
   */
  public function imageToImage(string|array|ImageToImageInput $input, string $model_id, array $tags = []): ImageToImageOutput {
    $this->loadClient();

    // Normalize input.
    if ($input instanceof ImageToImageInput) {
      $imageFile = $input->getImageFile();
      $prompt = $input->getPrompt();
      $mask = $input->getMask();
    }
    elseif (is_string($input)) {
      // Assume it's a file path.
      $imageFile = new ImageFile();
      $imageFile->setFileFromUri($input);
      $prompt = NULL;
      $mask = NULL;
    }
    else {
      throw new \InvalidArgumentException('Invalid input type for imageToImage operation.');
    }

    // Save image to temporary file for API upload.
    $imagePath = $this->fileSystem->saveData(
      $imageFile->getBinary(),
      'temporary://image_to_image_input.png',
      \Drupal\Core\File\FileExists::Replace
    );

    $payload = [
      'model' => $model_id,
      'image' => fopen($imagePath, 'r'),
    ];

    if ($prompt) {
      $payload['prompt'] = $prompt;
    }

    if ($mask) {
      $maskPath = $this->fileSystem->saveData(
        $mask->getBinary(),
        'temporary://image_to_image_mask.png',
        \Drupal\Core\File\FileExists::Replace
      );
      $payload['mask'] = fopen($maskPath, 'r');
    }

    $payload = $payload + $this->configuration;

    try {
      $response = $this->client->images()->edit($payload)->toArray();

      $images = [];
      foreach ($response['data'] as $data) {
        if (isset($data['b64_json'])) {
          $images[] = new ImageFile(base64_decode($data['b64_json']), 'image/png', 'edited.png');
        }
        elseif (isset($data['url']) && !empty($data['url'])) {
          try {
            $image_content = file_get_contents($data['url']);
            if ($image_content !== FALSE) {
              $images[] = new ImageFile($image_content, 'image/png', 'edited.png');
            }
          }
          catch (\Exception $e) {
            $this->loggerFactory->get('ai_provider_infomaniak')->error(
              'Failed to fetch image from URL @url: @message',
              ['@url' => $data['url'], '@message' => $e->getMessage()]
            );
          }
        }
      }

      if (empty($images)) {
        throw new \RuntimeException('No valid images were generated.');
      }

      return new ImageToImageOutput($images, $response, []);
    }
    catch (\Exception $e) {
      $this->handleApiException($e);
      throw $e;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function requiresImageToImageMask(string $model_id): bool {
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function hasImageToImageMask(string $model_id): bool {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function requiresImageToImagePrompt(string $model_id): bool {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function hasImageToImagePrompt(string $model_id): bool {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  protected function handleApiException(\Exception $e): void {
    $message = $e->getMessage();

    // Infomaniak-specific error patterns.
    if (str_contains($message, 'product_id') ||
        str_contains($message, 'invalid product')) {
      throw new AiSetupFailureException('Invalid Infomaniak Product ID: ' . $message);
    }

    // Rate limiting.
    if (str_contains($message, '429') ||
        str_contains($message, 'Too Many Requests') ||
        str_contains($message, 'rate limit')) {
      throw new AiRateLimitException('Infomaniak API rate limit exceeded: ' . $message);
    }

    // Quota exceeded.
    if (str_contains($message, 'quota') ||
        str_contains($message, 'insufficient credits') ||
        str_contains($message, 'billing') ||
        str_contains($message, 'exceeded your current quota')) {
      throw new AiQuotaException('Infomaniak API quota exceeded: ' . $message);
    }

    // Authentication failures.
    if (str_contains($message, '401') ||
        str_contains($message, 'Unauthorized') ||
        str_contains($message, 'invalid api key') ||
        str_contains($message, 'authentication')) {
      throw new AiSetupFailureException('Invalid Infomaniak API credentials: ' . $message);
    }

    // Call parent handler for common OpenAI errors.
    parent::handleApiException($e);
  }

  /**
   * {@inheritdoc}
   */
  public function getSetupData(): array {
    $setup = [];
    $setup['key_config_name'] = 'api_key';

    try {
      $models = $this->getConfiguredModels();

      if (!empty($models)) {
        // Set default chat model.
        $chat_models = $this->getConfiguredModels('chat');
        if (!empty($chat_models)) {
          $setup['default_models']['chat'] = array_key_first($chat_models);
        }

        // Set default text-to-image model.
        $image_models = $this->getConfiguredModels('text_to_image');
        if (!empty($image_models)) {
          $setup['default_models']['text_to_image'] = array_key_first($image_models);
        }

        // Set default speech-to-text model.
        $stt_models = $this->getConfiguredModels('speech_to_text');
        if (!empty($stt_models)) {
          $setup['default_models']['speech_to_text'] = array_key_first($stt_models);
        }

        // Set default image-to-image model.
        $i2i_models = $this->getConfiguredModels('image_to_image');
        if (!empty($i2i_models)) {
          $setup['default_models']['image_to_image'] = array_key_first($i2i_models);
        }

        // Set default embeddings model.
        $emb_models = $this->getConfiguredModels('embeddings');
        if (!empty($emb_models)) {
          $setup['default_models']['embeddings'] = array_key_first($emb_models);
        }
      }
    }
    catch (\Exception $e) {
      // If we can't get models, just return basic setup data.
      $this->loggerFactory->get('ai_provider_infomaniak')->warning(
        'Failed to get models for setup: @error',
        ['@error' => $e->getMessage()]
      );
    }

    return $setup;
  }

  /**
   * {@inheritdoc}
   */
  public function textToImage(string|TextToImageInput $input, string $model_id, array $tags = []): TextToImageOutput {
    // Images use API v1 endpoint - create temporary client.
    if (!$this->hasAuthentication()) {
      $this->loadAuthentication();
    }

    $endpoint = sprintf(
      'https://api.infomaniak.com/1/ai/%s/openai',
      $this->productId
    );

    // Create one-time client with API v1 endpoint.
    $imageClient = \OpenAI::factory()
      ->withApiKey($this->apiKey)
      ->withHttpClient($this->httpClient)
      ->withBaseUri($endpoint)
      ->make();

    if ($input instanceof TextToImageInput) {
      $input = $input->getText();
    }

    $payload = [
      'model' => $model_id,
      'prompt' => $input,
    ] + $this->configuration;

    try {
      $response = $imageClient->images()->create($payload)->toArray();
    }
    catch (\Exception $e) {
      $this->handleApiException($e);
      throw $e;
    }

    $images = [];
    if (empty($response['data'][0])) {
      throw new AiResponseErrorException('No image data found in the response.');
    }

    foreach ($response['data'] as $data) {
      if (isset($data['b64_json'])) {
        $images[] = new ImageFile(
          base64_decode($data['b64_json']),
          'image/png',
          'generated.png'
        );
      }
      elseif (isset($data['url']) && !empty($data['url'])) {
        try {
          $image_content = file_get_contents($data['url']);
          if ($image_content !== FALSE) {
            $images[] = new ImageFile($image_content, 'image/png', 'generated.png');
          }
        }
        catch (\Exception $e) {
          $this->loggerFactory->get('ai_provider_infomaniak')->error(
            'Failed to fetch image from URL @url: @message',
            ['@url' => $data['url'], '@message' => $e->getMessage()]
          );
        }
      }
    }

    if (empty($images)) {
      throw new AiResponseErrorException(
        'Failed to process any valid images from the API response.'
      );
    }

    return new TextToImageOutput($images, $response, []);
  }

}
