<?php

namespace Drupal\ai_provider_moonshot\Plugin\AiProvider;

use Drupal\ai\Exception\AiQuotaException;
use Drupal\ai\Exception\AiRateLimitException;
use Drupal\ai\Exception\AiResponseErrorException;
use Drupal\ai\OperationType\Embeddings\EmbeddingsInput;
use Drupal\ai\OperationType\Embeddings\EmbeddingsOutput;
use Drupal\ai\OperationType\GenericType\ImageFile;
use Drupal\ai\OperationType\TextToImage\TextToImageInput;
use Drupal\ai\OperationType\TextToImage\TextToImageOutput;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\AiProvider;
use Drupal\ai\Base\AiProviderClientBase;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatInterface;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\Chat\ChatOutput;
use Drupal\ai\OperationType\Chat\Tools\ToolsFunctionOutput;
use Drupal\ai\Traits\OperationType\ChatTrait;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the 'moonshot' provider.
 */
#[AiProvider(
  id: 'moonshot',
  label: new TranslatableMarkup('Moonshot AI'),
)]
class MoonshotProvider extends AiProviderClientBase implements
  ChatInterface
{

  use ChatTrait;

  /**
   * The Moonshot AI Client.
   *
   * @var \OpenAI\Client|null
   */
  protected $client;

  /**
   * API Key.
   *
   * @var string
   */
  protected string $apiKey = '';

  /**
   * Run moderation call, before a normal call.
   *
   * @var bool
   */
  protected bool $moderation = TRUE;

  /**
   * Retrieves the list of configured models supported by this provider.
   *
   * @param string|null $operation_type
   *   The operation type, e.g., "chat".
   * @param array $capabilities
   *   Specific capabilities to filter models by.
   *
   * @return array
   *   An array of supported model configurations.
   *
   * @throws \Drupal\ai\Exception\AiResponseErrorException
   *   Thrown if the models cannot be fetched.
   */
  public function getConfiguredModels(?string $operation_type = NULL, array $capabilities = []): array
  {
    $this->loadClient();
    switch ($operation_type) {
      case 'chat':
        $models = [
          'kimi-k2-0711-preview',
          'kimi-k2-turbo-preview',
          'kimi-latest',
          'kimi-thinking-preview',
        ];
        break;
      default:
        $models = [];
        break;
    }
    return array_combine($models, $models);
  }

  /**
   * {@inheritdoc}
   */
  public function isUsable(?string $operation_type = NULL, array $capabilities = []): bool
  {
    if (!$this->apiKey && !$this->getConfig()->get('api_key')) {
      return FALSE;
    }
    if ($operation_type) {
      return in_array($operation_type, $this->getSupportedOperationTypes());
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedOperationTypes(): array
  {
    return [
      'chat',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig(): ImmutableConfig
  {
    return $this->configFactory->get('ai_provider_moonshot.settings');
  }

//  /**
//   * {@inheritdoc}
//   */
//  public function getApiDefinition(): array
//  {
//    $definition = Yaml::parseFile($this->moduleHandler->getModule('ai_provider_moonshot')->getPath() . '/definitions/api_defaults.yml');
//    return $definition;
//  }

  /**
   * {@inheritdoc}
   */
  public function getModelSettings(string $model_id, array $generalConfig = []): array
  {
    return $generalConfig;
  }

  /**
   * {@inheritdoc}
   */
  public function setAuthentication(mixed $authentication): void
  {
    $this->apiKey = $authentication;
    $this->client = NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function chat(array|string|ChatInput $input, string $model_id, array $tags = []): ChatOutput
  {
    $this->loadClient();
    $chat_input = $input;
    if ($input instanceof ChatInput) {
      $chat_input = [];
      if ($this->chatSystemRole) {
        $chat_input[] = [
          'role' => 'system',
          'content' => $this->chatSystemRole,
        ];
      }
      foreach ($input->getMessages() as $message) {
        $content = [
          [
            'type' => 'text',
            'text' => $message->getText(),
          ],
        ];
        if (count($message->getImages())) {
          foreach ($message->getImages() as $image) {
            $content[] = [
              'type' => 'image_url',
              'image_url' => [
                'url' => $image->getAsBase64EncodedString(),
              ],
            ];
          }
        }
        $new_message = [
          'role' => $message->getRole(),
          'content' => $content,
        ];

        if ($message->getRole() == 'tool') {
          $new_message = [
            'role' => 'tool',
            'content' => $message->getText(),
          ];
        }

        if ($message->getToolsId()) {
          $new_message['tool_call_id'] = $message->getToolsId();
        }

        if ($message->getTools()) {
          $new_message['tool_calls'] = $message->getRenderedTools();
        }

        $chat_input[] = $new_message;
      }
    }

    $payload = [
        'model' => $model_id,
        'messages' => $chat_input,
      ] + $this->configuration;

    if (method_exists($input, 'getChatTools') && $input->getChatTools()) {
      $payload['tools'] = $input->getChatTools()->renderToolsArray();
      foreach ($payload['tools'] as $key => $tool) {
        $payload['tools'][$key]['function']['strict'] = FALSE;
      }
    }

    if (method_exists($input, 'getChatStructuredJsonSchema') && $input->getChatStructuredJsonSchema()) {
      $payload['response_format'] = [
        'type' => 'json_schema',
        'json_schema' => $input->getChatStructuredJsonSchema(),
      ];
    }

    try {
      $response = $this->client->chat()->create($payload)->toArray();

      $tools = [];
      if (!empty($response['choices'][0]['message']['tool_calls'])) {
        foreach ($response['choices'][0]['message']['tool_calls'] as $tool) {
          $arguments = Json::decode($tool['function']['arguments']);
          $tools[] = new ToolsFunctionOutput($input->getChatTools()->getFunctionByName($tool['function']['name']), $tool['id'], $arguments);
        }
      }
      $message = new ChatMessage($response['choices'][0]['message']['role'], $response['choices'][0]['message']['content'] ?? "", []);
      if (!empty($tools)) {
        $message->setTools($tools);
      }
    } catch (\Exception $e) {
      throw $e;
    }

    return new ChatOutput($message, $response, []);
  }

  /**
   * {@inheritdoc}
   */
  public function getSetupData(): array
  {
    return [
      'key_config_name' => 'api_key',
      'default_models' => [
        'chat' => 'moonshot-lite',
        'chat_with_image_vision' => 'moonshot-pro',
        'chat_with_complex_json' => 'moonshot-pro',
        'chat_with_tools' => 'moonshot-pro',
        'chat_with_structured_response' => 'moonshot-pro',
      ],
    ];
  }

  /**
   * Loads the Moonshot AI Client with authentication if not initialized.
   */
  protected function loadClient(): void
  {
    if (!$this->client) {
      if (!$this->apiKey) {
        $this->setAuthentication($this->loadApiKey());
      }
      $host = $this->config->get('host');
      $client = \OpenAI::factory()
        ->withApiKey($this->apiKey)
        ->withBaseUri($host)
        ->withHttpClient($this->httpClient);

      if (!empty($this->getConfig()->get('host'))) {
        $client->withBaseUri($this->getConfig()->get('host'));
      }

      $this->client = $client->make();
    }
  }
}
