<pre>
  ┌───────┐
  │       │
  │  a:o  │  acolono.com
  │       │
  └───────┘
</pre>

# NanoBanana Provider (Google Gemini Image Models)

AI Provider module for Google Gemini's image generation models (Gemini 2.5 Flash Image and Gemini 3 Pro Image).

## Overview

NanoBanana is an AI provider plugin for the [Drupal AI module](https://www.drupal.org/project/ai) that integrates Google's Gemini image generation models. It supports both text-to-image and image-to-image generation with advanced features like multi-image composition, aspect ratio control, and high-resolution output.

## Features

- **Text-to-Image Generation**: Generate images from text descriptions
- **Image-to-Image Transformation**: Edit and transform existing images
- **Multi-Image Composition**: Combine up to 14 images into composite scenes (model-dependent)
- **Aspect Ratio Control**: 10 aspect ratio options (1:1, 3:4, 4:3, 9:16, 16:9, etc.)
- **High-Resolution Output**: Up to 4K resolution with Gemini 3 Pro
- **AI API Explorer Integration**: Built-in UI for testing with automatic multi-image field injection

## Supported Models

### Gemini 2.5 Flash Image
- **Speed**: Fast, low-latency
- **Multi-Image**: Up to 3 images (1 main + 2 additional)
- **Best For**: Quick iterations, prototyping, real-time applications

### Gemini 3 Pro Image (Preview)
- **Speed**: Slower, higher quality
- **Multi-Image**: Up to 14 images (1 main + 13 additional)
- **Resolution**: 1K, 2K, 4K
- **Best For**: Final production, high-quality output, complex compositions

## Requirements

- Drupal 10.3+ or Drupal 11+
- [AI module](https://www.drupal.org/project/ai)
- [Key module](https://www.drupal.org/project/key)
- Google Gemini API key ([Get one here](https://aistudio.google.com/apikey))

## Installation

1. Install dependencies:
   ```bash
   composer require drupal/ai drupal/key
   ```

2. Enable the required modules:
   ```bash
   drush en ai key ai_provider_nanobanana
   ```

3. Clear cache:
   ```bash
   drush cr
   ```

## Configuration

### 1. Configure API Key

Use the Key module to store your API key securely:
1. Navigate to **Configuration → System → Keys**
2. Add a new key with your Gemini API key
3. Select it in the NanoBanana settings (`/admin/config/ai/providers/nanobanana`)

### 2. Verify Configuration

Navigate to **AI API Explorer → Image-To-Image Explorer** (`/admin/config/ai/explorers/image_to_image_generator`):
- Select **Provider**: NanoBanana
- The available models should appear in the dropdown

## Usage

### AI API Explorer

#### Text-to-Image
1. Navigate to **AI API Explorer → Text-To-Image Explorer**
2. Select **Provider**: NanoBanana
3. Select **Model**: Gemini 2.5 Flash Image or Gemini 3 Pro Image
4. Configure options:
   - **Aspect Ratio**: Choose from 10 options
   - **Image Size**: (Gemini 3 Pro only) Default, 1K, 2K, or 4K
5. Enter your prompt
6. Click **Generate Image**

#### Image-to-Image
1. Navigate to **AI API Explorer → Image-To-Image Explorer**
2. Select **Provider**: NanoBanana
3. Select **Model**: Gemini 2.5 Flash Image or Gemini 3 Pro Image
4. Upload main image
5. **Multi-Image**: Upload additional images (fields appear automatically)
6. Configure options and enter prompt
7. Click **Generate Image**

### Programmatic Usage

#### Simple Text-to-Image

```php
use Drupal\ai\OperationType\TextToImage\TextToImageInput;

// Create the provider
$provider = \Drupal::service('ai.provider')->createInstance('nanobanana');

// Configure aspect ratio
$provider->setConfiguration(['aspectRatio' => '16:9']);

// Create input
$input = new TextToImageInput('A serene mountain landscape at sunset');

// Generate
$response = $provider->textToImage(
  $input,
  'gemini-2.5-flash-image',
  ['my_module']
);

// Get result
$images = $response->getNormalized();
$binary = $images[0]->getBinary();

// Save as file
$file = $images[0]->getAsFileEntity('public://', 'generated-image.png');

// Or save as media
$media = $images[0]->getAsMediaEntity('image', '', 'generated-image.png');
```

#### Image-to-Image

```php
use Drupal\ai\OperationType\GenericType\ImageFile;
use Drupal\ai\OperationType\ImageToImage\ImageToImageInput;

// Load image
$image_data = file_get_contents('path/to/image.jpg');
$image = new ImageFile($image_data, 'image/jpeg', 'image.jpg');

// Create provider
$provider = \Drupal::service('ai.provider')->createInstance('nanobanana');

// Create input
$input = new ImageToImageInput($image);
$input->setPrompt('Convert this to a watercolor painting');

// Generate
$response = $provider->imageToImage(
  $input,
  'gemini-2.5-flash-image',
  ['my_module']
);

$images = $response->getNormalized();
```

#### Multi-Image Composition

```php
use Drupal\ai\OperationType\GenericType\ImageFile;
use Drupal\ai_provider_nanobanana\OperationType\MultiImageToImageInput;

// Load images
$image1_data = file_get_contents('image1.jpg');
$image2_data = file_get_contents('image2.jpg');
$image3_data = file_get_contents('image3.jpg');

// Create ImageFile objects
$image1 = new ImageFile($image1_data, 'image/jpeg', 'image1.jpg');
$image2 = new ImageFile($image2_data, 'image/jpeg', 'image2.jpg');
$image3 = new ImageFile($image3_data, 'image/jpeg', 'image3.jpg');

// Create provider
$provider = \Drupal::service('ai.provider')->createInstance('nanobanana');
$provider->setConfiguration(['aspectRatio' => '16:9']);

// Create multi-image input
$input = new MultiImageToImageInput($image1);
$input->setPrompt('Combine these images into a single cohesive scene');
$input->addAdditionalImage($image2);
$input->addAdditionalImage($image3);

// Generate (use Gemini 3 Pro for more images)
$response = $provider->imageToImage(
  $input,
  'gemini-3-pro-image-preview',
  ['my_module']
);

$images = $response->getNormalized();
```

#### High-Resolution Output (Gemini 3 Pro)

```php
use Drupal\ai\OperationType\TextToImage\TextToImageInput;

$provider = \Drupal::service('ai.provider')->createInstance('nanobanana');

// Configure for 4K output
$provider->setConfiguration([
  'aspectRatio' => '16:9',
  'imageSize' => '4K',
]);

$input = new TextToImageInput('A highly detailed cityscape at night');

$response = $provider->textToImage(
  $input,
  'gemini-3-pro-image-preview',
  ['my_module']
);

$images = $response->getNormalized();
```

## Configuration Options

### Aspect Ratio
Controls the aspect ratio of generated images:
- `1:1` - Square
- `2:3`, `3:2` - Photo formats
- `3:4`, `4:3` - Standard formats
- `4:5`, `5:4` - Portrait/landscape
- `9:16`, `16:9` - Vertical/widescreen
- `21:9` - Ultra-wide

### Image Size (Gemini 3 Pro Only)
Controls output resolution:
- `` (empty) - Default resolution
- `1K` - ~1000px
- `2K` - ~2000px
- `4K` - ~4000px

**Note**: Higher resolutions take longer to generate.

## Multi-Image Feature

The module supports multi-image composition where multiple images can be provided as context for generation.

**Limits by Model:**
- **Gemini 2.5 Flash**: Up to 3 total images (1 main + 2 additional)
- **Gemini 3 Pro**: Up to 14 total images (1 main + 13 additional)

**Use Cases:**
- Composite scenes and photo collages
- Style transfer between images
- Character and object combinations
- Scene blending and merging

## Integration with AI API Explorer

When you select NanoBanana as the provider in the AI API Explorer's Image-To-Image interface, the module automatically:

1. **Injects Multi-Image Fields**: Additional file upload fields appear dynamically
2. **Shows Thumbnail Previews**: Selected images display as thumbnails with labels
3. **Validates Image Count**: Enforces model-specific limits with helpful error messages
4. **Generates Code Examples**: Provides copy-paste PHP code for your implementation

## Troubleshooting

### "NO_IMAGE" Error
The AI model didn't generate an image. This can happen when:
- The prompt is too vague or unclear
- The request conflicts with content policies
- **Solution**: Be more specific in your prompt with clear, descriptive language

### API Key Issues
- Verify your API key at https://aistudio.google.com/apikey
- Ensure the key has the correct permissions
- Check that the key is properly configured in the module settings

### Image Upload Errors
- **"Too many images"**: Reduce the number of images to match the model's limit
- **File size issues**: Ensure images are reasonable size (<10MB recommended)
- **Format issues**: Use standard formats (JPEG, PNG)

### Performance Issues
- Use Gemini 2.5 Flash for faster generation
- Lower resolution settings generate faster
- Fewer images in multi-image mode process faster

## Resources

- [Google Gemini API Documentation](https://ai.google.dev/gemini-api/docs/image-generation)
- [Drupal AI Module Documentation](https://www.drupal.org/project/ai)
- [Get Gemini API Key](https://aistudio.google.com/apikey)

by acolono GmbH
---------------

~~we build your websites~~
we build your business

hello@acolono.com

www.acolono.com
www.twitter.com/acolono
www.drupal.org/acolono-gmbh

