(function (Drupal) {
  'use strict';

  Drupal.behaviors.nanoBananaImagePreview = {
    attach: function (context, settings) {
      console.log('NanoBanana preview behavior attached to context');

      // Find all file inputs with 'additional_images' in the name.
      const fileInputs = context.querySelectorAll('input[type="file"]');
      console.log('Found ' + fileInputs.length + ' file inputs in context');

      fileInputs.forEach(function(fileInput) {
        console.log('Checking file input:', fileInput.name);

        if (!fileInput.name || !fileInput.name.includes('additional_images')) {
          return;
        }

        // Only attach once.
        if (fileInput.dataset.nanobananaPreviewed) {
          console.log('Already attached to this input');
          return;
        }
        fileInput.dataset.nanobananaPreviewed = 'true';

        console.log('Setting up preview for:', fileInput.name);

        // Create or find preview container.
        let previewContainer = fileInput.parentElement.querySelector('.nanobanana-image-previews');
        if (!previewContainer) {
          previewContainer = document.createElement('div');
          previewContainer.className = 'nanobanana-image-previews';
          fileInput.parentElement.insertBefore(previewContainer, fileInput.nextSibling);
        }

        // Handle file selection.
        fileInput.addEventListener('change', function(e) {
          console.log('File input changed, files selected:', e.target.files.length);

          // Clear previous previews.
          previewContainer.innerHTML = '';

          const files = e.target.files;
          if (files.length === 0) {
            console.log('No files selected');
            return;
          }

          // Show count.
          const countLabel = document.createElement('div');
          countLabel.className = 'nanobanana-preview-count';
          countLabel.textContent = files.length + ' image' + (files.length > 1 ? 's' : '') + ' selected:';
          previewContainer.appendChild(countLabel);

          // Create thumbnails wrapper.
          const thumbsWrapper = document.createElement('div');
          thumbsWrapper.className = 'nanobanana-preview-thumbnails';
          previewContainer.appendChild(thumbsWrapper);

          // Generate preview for each file.
          Array.from(files).forEach(function(file, index) {
            if (!file.type.match('image.*')) {
              return;
            }

            const reader = new FileReader();

            reader.onload = function(e) {
              const thumbContainer = document.createElement('div');
              thumbContainer.className = 'nanobanana-preview-thumb';

              const img = document.createElement('img');
              img.src = e.target.result;
              img.alt = file.name;
              img.title = file.name;

              const label = document.createElement('div');
              label.className = 'nanobanana-preview-label';
              label.textContent = (index + 1) + '. ' + file.name;

              thumbContainer.appendChild(img);
              thumbContainer.appendChild(label);
              thumbsWrapper.appendChild(thumbContainer);
            };

            reader.readAsDataURL(file);
          });
        });
      });
    }
  };

})(Drupal);
