<?php

namespace Drupal\ai_provider_nanobanana;

use Drupal\ai_provider_nanobanana\Form\NanoBananaConfigForm;
use Drupal\Core\Config\ConfigFactory;
use Drupal\key\KeyRepositoryInterface;
use GuzzleHttp\Client;

/**
 * NanoBanana (Gemini 2.5 Flash Image) API client.
 */
class NanoBanana {

  /**
   * The http client.
   */
  protected Client $client;

  /**
   * API Key.
   */
  private string $apiKey;

  /**
   * The base path.
   */
  private string $basePath = 'https://generativelanguage.googleapis.com/';

  /**
   * The model ID.
   */
  private string $modelId = 'gemini-2.5-flash-image';

  /**
   * Constructs a new NanoBanana object.
   *
   * @param \GuzzleHttp\Client $client
   *   Http client.
   * @param \Drupal\Core\Config\ConfigFactory $configFactory
   *   The config factory.
   * @param \Drupal\key\KeyRepositoryInterface $keyRepository
   *   The key repository.
   */
  public function __construct(Client $client, ConfigFactory $configFactory, KeyRepositoryInterface $keyRepository) {
    $this->client = $client;
    $key = $configFactory->get(NanoBananaConfigForm::CONFIG_NAME)->get('api_key') ?? '';
    if ($key) {
      $this->apiKey = $keyRepository->getKey($key)->getKeyValue();
    }
  }

  /**
   * Set API key.
   *
   * @param string $key
   *   The API key to set.
   */
  public function setApiKey($key) {
    $this->apiKey = $key;
  }

  /**
   * Generate image from text.
   *
   * @param string $prompt
   *   The prompt to use.
   * @param string $model_id
   *   The model ID to use.
   * @param array $options
   *   Other options to add, like aspectRatio, imageSize, etc.
   *
   * @return array
   *   Array of image binaries.
   */
  public function generateImage($prompt, $model_id = 'gemini-2.5-flash-image', $options = []) {
    $contents = [
      [
        'parts' => [
          ['text' => $prompt],
        ],
      ],
    ];

    $config = $this->buildGenerationConfig($model_id, $options);

    $data = [
      'contents' => $contents,
      'generationConfig' => $config,
    ];

    $response = $this->makeRequest("v1beta/models/{$model_id}:generateContent", [], 'POST', json_encode($data));

    return $this->extractImagesFromResponse($response);
  }

  /**
   * Generate image from image and optional text.
   *
   * @param string $imageBinary
   *   The input image binary.
   * @param string $prompt
   *   The prompt to use (optional for image-to-image).
   * @param string $model_id
   *   The model ID to use.
   * @param array $options
   *   Other options to add, like aspectRatio, imageSize, etc.
   *
   * @return array
   *   Array of image binaries.
   */
  public function generateImageFromImage($imageBinary, $prompt = '', $model_id = 'gemini-2.5-flash-image', $options = []) {
    // Encode the image to base64.
    $imageBase64 = base64_encode($imageBinary);

    // Detect mime type from binary data.
    $finfo = new \finfo(FILEINFO_MIME_TYPE);
    $mimeType = $finfo->buffer($imageBinary);

    $parts = [
      [
        'inline_data' => [
          'mime_type' => $mimeType,
          'data' => $imageBase64,
        ],
      ],
    ];

    // Add text prompt if provided.
    if (!empty($prompt)) {
      $parts[] = ['text' => $prompt];
    }

    $contents = [
      [
        'parts' => $parts,
      ],
    ];

    $config = $this->buildGenerationConfig($model_id, $options);

    $data = [
      'contents' => $contents,
      'generationConfig' => $config,
    ];

    $response = $this->makeRequest("v1beta/models/{$model_id}:generateContent", [], 'POST', json_encode($data));

    return $this->extractImagesFromResponse($response);
  }

  /**
   * Generate image from multiple images and optional text.
   *
   * @param array $imageBinaries
   *   Array of input image binaries (up to 3 images for Flash, 14 for Pro).
   * @param string $prompt
   *   The prompt to use to describe what to create from the images.
   * @param string $model_id
   *   The model ID to use.
   * @param array $options
   *   Other options to add, like aspectRatio, imageSize, etc.
   *
   * @return array
   *   Array of image binaries.
   */
  public function generateImageFromImages(array $imageBinaries, $prompt = '', $model_id = 'gemini-2.5-flash-image', $options = []) {
    $parts = [];

    // Add each image to the parts array.
    foreach ($imageBinaries as $imageBinary) {
      // Encode the image to base64.
      $imageBase64 = base64_encode($imageBinary);

      // Detect mime type from binary data.
      $finfo = new \finfo(FILEINFO_MIME_TYPE);
      $mimeType = $finfo->buffer($imageBinary);

      $parts[] = [
        'inline_data' => [
          'mime_type' => $mimeType,
          'data' => $imageBase64,
        ],
      ];
    }

    // Add text prompt if provided.
    if (!empty($prompt)) {
      $parts[] = ['text' => $prompt];
    }

    $contents = [
      [
        'parts' => $parts,
      ],
    ];

    $config = $this->buildGenerationConfig($model_id, $options);

    $data = [
      'contents' => $contents,
      'generationConfig' => $config,
    ];

    $response = $this->makeRequest("v1beta/models/{$model_id}:generateContent", [], 'POST', json_encode($data));

    return $this->extractImagesFromResponse($response);
  }

  /**
   * Build generation config from options.
   *
   * @param string $model_id
   *   The model ID being used.
   * @param array $options
   *   The options array.
   *
   * @return array
   *   The generation config.
   */
  protected function buildGenerationConfig($model_id, array $options) {
    // Gemini 3 Pro requires TEXT and IMAGE modalities.
    if ($model_id === 'gemini-3-pro-image-preview') {
      $config = [
        'responseModalities' => ['TEXT', 'IMAGE'],
      ];
    }
    else {
      $config = [
        'responseModalities' => ['Image'],
      ];
    }

    // Build image config for aspect ratio and image size.
    $imageConfig = [];
    if (!empty($options['aspectRatio'])) {
      $imageConfig['aspectRatio'] = $options['aspectRatio'];
    }

    // imageSize is only for Gemini 3 Pro (1K, 2K, 4K - must be uppercase).
    if ($model_id === 'gemini-3-pro-image-preview' && !empty($options['imageSize'])) {
      $imageConfig['imageSize'] = strtoupper($options['imageSize']);
    }

    if (!empty($imageConfig)) {
      $config['imageConfig'] = $imageConfig;
    }

    return $config;
  }

  /**
   * Extract images from API response.
   *
   * @param string $responseBody
   *   The response body.
   *
   * @return array
   *   Array of image binaries.
   */
  protected function extractImagesFromResponse($responseBody) {
    $response = json_decode($responseBody, TRUE);

    if (!isset($response['candidates']) || !is_array($response['candidates'])) {
      throw new \Exception('No candidates found in response. Response: ' . $responseBody);
    }

    $images = [];
    foreach ($response['candidates'] as $candidate) {
      if (isset($candidate['content']['parts']) && is_array($candidate['content']['parts'])) {
        foreach ($candidate['content']['parts'] as $part) {
          // Look for inline_data with image data.
          if (isset($part['inline_data']['data'])) {
            // Decode base64 image data.
            $images[] = base64_decode($part['inline_data']['data']);
          }
          // Also check for inlineData (camelCase variant).
          elseif (isset($part['inlineData']['data'])) {
            $images[] = base64_decode($part['inlineData']['data']);
          }
        }
      }
    }

    if (empty($images)) {
      throw new \Exception('No valid image data found in response. Response: ' . $responseBody);
    }

    return $images;
  }

  /**
   * Make API call to Gemini.
   *
   * @param string $path
   *   The path.
   * @param array $query_string
   *   The query string.
   * @param string $method
   *   The method.
   * @param string $body
   *   Data to attach if POST/PUT/PATCH.
   * @param array $options
   *   Extra options.
   *
   * @return string
   *   The return response body.
   */
  protected function makeRequest($path, array $query_string = [], $method = 'GET', $body = '', array $options = []) {
    // Check so the api key exists.
    if (!$this->apiKey) {
      throw new \Exception('The Google Gemini API key has not been set.');
    }

    // Add API key to query string.
    $query_string['key'] = $this->apiKey;

    // Set timeouts (increased for multi-image processing).
    $options['timeout'] = 180;
    $options['connect_timeout'] = 60;
    $options['read_timeout'] = 180;

    // Don't let Guzzle die, just forward body and status.
    $options['http_errors'] = FALSE;

    // Set headers.
    $options['headers']['Content-Type'] = 'application/json';

    if ($body) {
      $options['body'] = $body;
    }

    $url = $this->basePath . $path;
    $url .= count($query_string) ? '?' . http_build_query($query_string) : '';

    $res = $this->client->request($method, $url, $options);

    // Get the response code.
    $code = $res->getStatusCode();

    // Fail on 4xx and 5xx.
    if (in_array(substr($code, 0, 1), ['4', '5'])) {
      $errorBody = $res->getBody()->getContents();
      throw new \Exception('The Google Gemini API returned an error: ' . $errorBody . ' with code ' . $code);
    }

    return $res->getBody()->getContents();
  }

}
