<?php

namespace Drupal\ai_provider_nanobanana\Plugin\AiProvider;

use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\AiProvider;
use Drupal\ai\Base\AiProviderClientBase;
use Drupal\ai\Exception\AiResponseErrorException;
use Drupal\ai\OperationType\GenericType\ImageFile;
use Drupal\ai\OperationType\ImageToImage\ImageToImageInput;
use Drupal\ai\OperationType\ImageToImage\ImageToImageInterface;
use Drupal\ai\OperationType\ImageToImage\ImageToImageOutput;
use Drupal\ai\OperationType\TextToImage\TextToImageInput;
use Drupal\ai\OperationType\TextToImage\TextToImageInterface;
use Drupal\ai\OperationType\TextToImage\TextToImageOutput;
use Drupal\ai\Traits\OperationType\ImageToImageTrait;
use Drupal\ai_provider_nanobanana\NanoBanana;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the 'nanobanana' provider.
 */
#[AiProvider(
  id: 'nanobanana',
  label: new TranslatableMarkup('NanoBanana'),
)]
class NanoBananaProvider extends AiProviderClientBase implements
  ContainerFactoryPluginInterface,
  ImageToImageInterface,
  TextToImageInterface {

  use ImageToImageTrait;

  /**
   * The NanoBanana Client.
   *
   * @var \Drupal\ai_provider_nanobanana\NanoBanana
   */
  protected $client;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The temporary files.
   *
   * @var array
   */
  protected $temporaryFiles = [];

  /**
   * Destructor.
   */
  public function __destruct() {
    foreach ($this->temporaryFiles as $file) {
      $file->delete();
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->client = $container->get('nanobanana.api');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguredModels(?string $operation_type = NULL, array $capabilities = []): array {
    $models = [];
    if (is_null($operation_type) || in_array($operation_type, ['text_to_image', 'image_to_image'])) {
      $models['gemini-2.5-flash-image'] = 'Gemini 2.5 Flash Image - Fast, low-latency';
      $models['gemini-3-pro-image-preview'] = 'Gemini 3 Pro Image - High quality, 4K support';
    }
    return $models;
  }

  /**
   * {@inheritdoc}
   */
  public function isUsable(?string $operation_type = NULL, array $capabilities = []): bool {
    // If its not configured, it is not usable.
    if (!$this->getConfig()->get('api_key')) {
      return FALSE;
    }
    // If its one of the operations that NanoBanana supports its usable.
    if ($operation_type) {
      return in_array($operation_type, $this->getSupportedOperationTypes());
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedOperationTypes(): array {
    return [
      'text_to_image',
      'image_to_image',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig(): ImmutableConfig {
    return $this->configFactory->get('ai_provider_nanobanana.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function getApiDefinition(): array {
    // Load the configuration.
    $definition = Yaml::parseFile($this->moduleHandler->getModule('ai_provider_nanobanana')->getPath() . '/definitions/api_defaults.yml');
    return $definition;
  }

  /**
   * {@inheritdoc}
   */
  public function getModelSettings(string $model_id, array $generalConfig = []): array {
    // Remove imageSize configuration for gemini-2.5-flash-image.
    if ($model_id === 'gemini-2.5-flash-image' && isset($generalConfig['imageSize'])) {
      unset($generalConfig['imageSize']);
    }

    return $generalConfig;
  }

  /**
   * {@inheritdoc}
   */
  public function setAuthentication(mixed $authentication): void {
    // Set the new API key and reset the client.
    $this->client->setApiKey($authentication);
  }

  /**
   * {@inheritdoc}
   */
  public function textToImage(string|TextToImageInput $input, string $model_id, array $tags = []): TextToImageOutput {
    // Normalize the input if needed.
    if ($input instanceof TextToImageInput) {
      $input = $input->getText();
    }

    try {
      $imageBinaries = $this->client->generateImage($input, $model_id, $this->configuration);

      if (empty($imageBinaries)) {
        throw new AiResponseErrorException('No image data found in the response.');
      }

      $images = [];
      $allBinaries = '';
      foreach ($imageBinaries as $index => $binary) {
        $filename = 'nanobanana_' . $index . '.png';
        $mimetype = 'image/png';
        $images[] = new ImageFile($binary, $mimetype, $filename);
        $allBinaries .= $binary;
      }

      return new TextToImageOutput($images, $allBinaries, []);
    }
    catch (\Exception $e) {
      throw new AiResponseErrorException('NanoBanana API error: ' . $e->getMessage());
    }
  }

  /**
   * {@inheritdoc}
   */
  public function imageToImage(string|array|ImageToImageInput $input, string $model_id, array $tags = []): ImageToImageOutput {
    if (!($input instanceof ImageToImageInput)) {
      throw new \InvalidArgumentException('Input must be an instance of ImageToImageInput for NanoBanana.');
    }

    $prompt = $input->getPrompt() ?? '';

    try {
      // Check if this is a MultiImageToImageInput with additional images.
      if ($input instanceof \Drupal\ai_provider_nanobanana\OperationType\MultiImageToImageInput) {
        $allImages = $input->getAllImages();
        $imageBinaries = [];
        foreach ($allImages as $imageFile) {
          $imageBinaries[] = $imageFile->getBinary();
        }
        $imageBinaries = $this->client->generateImageFromImages($imageBinaries, $prompt, $model_id, $this->configuration);
      }
      else {
        // Single image mode.
        $image = $input->getImageFile();
        $imageBinaries = $this->client->generateImageFromImage($image->getBinary(), $prompt, $model_id, $this->configuration);
      }

      if (empty($imageBinaries)) {
        throw new AiResponseErrorException('No image data found in the response.');
      }

      $images = [];
      $allBinaries = '';
      foreach ($imageBinaries as $index => $binary) {
        $filename = 'nanobanana_' . $index . '.png';
        $mimetype = 'image/png';
        $images[] = new ImageFile($binary, $mimetype, $filename);
        $allBinaries .= $binary;
      }

      return new ImageToImageOutput($images, $allBinaries, $tags);
    }
    catch (\Exception $e) {
      throw new AiResponseErrorException('NanoBanana API error: ' . $e->getMessage());
    }
  }

  /**
   * {@inheritdoc}
   */
  public function requiresImageToImageMask(string $model_id): bool {
    // NanoBanana does not require masks.
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function hasImageToImageMask(string $model_id): bool {
    // NanoBanana does not use masks.
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function requiresImageToImagePrompt(string $model_id): bool {
    // NanoBanana does not require prompts for image-to-image.
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function hasImageToImagePrompt(string $model_id): bool {
    // NanoBanana supports prompts for image-to-image.
    return TRUE;
  }

  /**
   * Gets the raw client.
   *
   * This is the client for inference.
   *
   * @return \Drupal\ai_provider_nanobanana\NanoBanana
   *   The client.
   */
  public function getClient(): NanoBanana {
    return $this->client;
  }

}
