<?php

namespace Drupal\ai_provider_siliconflow\Plugin\AiProvider;

use Drupal\ai\Exception\AiQuotaException;
use Drupal\ai\OperationType\GenericType\ImageFile;
use Drupal\ai\OperationType\TextToImage\TextToImageInput;
use Drupal\ai\OperationType\TextToImage\TextToImageOutput;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\AiProvider;
use Drupal\ai\Base\AiProviderClientBase;
use Drupal\ai\Exception\AiMissingFeatureException;
use Drupal\ai\Exception\AiRateLimitException;
use Drupal\ai\Exception\AiResponseErrorException;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatInterface;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\Chat\ChatOutput;
use Drupal\ai\OperationType\Embeddings\EmbeddingsInput;
use Drupal\ai\OperationType\Embeddings\EmbeddingsInterface;
use Drupal\ai\OperationType\Embeddings\EmbeddingsOutput;
use Drupal\ai\OperationType\ImageClassification\ImageClassificationInput;
use Drupal\ai\OperationType\ImageClassification\ImageClassificationInterface;
use Drupal\ai\OperationType\ImageClassification\ImageClassificationItem;
use Drupal\ai\OperationType\ImageClassification\ImageClassificationOutput;
use Drupal\ai\Traits\OperationType\ChatTrait;
use Drupal\ai_provider_siliconflow\SiliconflowApi;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the 'siliconflow' provider.
 */
#[AiProvider(
  id: 'siliconflow',
  label: new TranslatableMarkup('Siliconflow'),
)]
class SiliconflowProvider extends AiProviderClientBase implements
  ContainerFactoryPluginInterface,
  ChatInterface,
  EmbeddingsInterface,
  ImageClassificationInterface {

  use StringTranslationTrait;
  use ChatTrait;

  /**
   * The Siliconflow Client.
   *
   * @var \Drupal\ai_provider_siliconflow\SiliconflowApi
   */
  protected SiliconflowApi $client;

  /**
   * API Key.
   *
   * @var string
   */
  protected string $apiKey = '';

  /**
   * We want to add models to the provider dynamically.
   *
   * @var bool
   */
  protected bool $hasPredefinedModels = FALSE;

  /**
   * Supported Types.
   *
   * @var array
   */
  protected $supportedTypes = [
    'chat' => [
      'label' => 'Chat',
      'filter' => 'chat',
    ],
    'embeddings' => [
      'label' => 'Embeddings',
      'filter' => 'embedding',
    ],
    'text_to_image' => [
      'label' => 'Text To Image',
      'filter' => 'text-to-image',
    ],
//    'image_classification' => [
//      'label' => 'Image Classification',
//      'filter' => 'image-classification',
//    ],
  ];

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->client = $container->get('ai_provider_siliconflow.api');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function isUsable(?string $operation_type = NULL, array $capabilities = []): bool {
    // If its not configured, it is not usable.
    if (!$this->getConfig()->get('api_key')) {
      return FALSE;
    }
    // If its one of the bundles that Mistral supports its usable.
    if ($operation_type) {
      return in_array($operation_type, $this->getSupportedOperationTypes());
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedOperationTypes(): array {
    return [
      'chat',
      'embeddings',
      'text_to_image',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig(): ImmutableConfig {
    return $this->configFactory->get('ai_provider_siliconflow.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function getApiDefinition(): array {
    // Load the configuration.
    $definition = Yaml::parseFile($this->moduleHandler->getModule('ai_provider_siliconflow')->getPath() . '/definitions/api_defaults.yml');
    return $definition;
  }

  /**
   * {@inheritdoc}
   */
  public function getModelSettings(string $model_id, array $generalConfig = []): array {
    return $generalConfig;
  }

  /**
   * {@inheritdoc}
   */
  public function setAuthentication(mixed $authentication): void {
    // Set the new API key and reset the client.
    $this->apiKey = $authentication;
    $this->client->setApiToken($this->apiKey);
  }

  /**
   * Gets the raw client.
   *
   * This is the client for inference.
   *
   * @return \Drupal\ai_provider_siliconflow\SiliconflowApi
   *   The Siliconflow client.
   */
  public function getClient(): SiliconflowApi {
    $this->loadClient();
    return $this->client;
  }

  /**
   * Loads the Siliconflow Client with authentication if not initialized.
   */
  protected function loadClient(): void {
    if (!$this->apiKey) {
      $this->setAuthentication($this->loadApiKey());
    }
    $this->client->setApiToken($this->apiKey);
  }

  /**
   * {@inheritdoc}
   */
  public function chat(array|string|ChatInput $input, string $model_id, array $tags = []): ChatOutput {
    $info = $this->getModelInfo('chat', $model_id);
    if (!$info['siliconflow_endpoint']) {
      throw new AiMissingFeatureException('Siliconflow endpoint is missing.');
    }
    $this->loadClient();
    // Normalize the input if needed.
    $chat_input = $input;
    if ($input instanceof ChatInput) {
      $chat_input = "";
      // Add a warning log if they set an system role.
      if ($this->chatSystemRole) {
        $this->loggerFactory->get('ai')->warning('A chat message with system role was sent with Siliconflow provider. Siliconflow does not support system roles and this was removed.');
      }
      foreach ($input->getMessages() as $message) {
        $chat_input .= $message->getRole() . ': ' . $message->getText() . "\n";
        if (count($message->getImages())) {
          throw new AiMissingFeatureException('Images are not supported by Siliconflow.');
        }
      }
    }
    try {
      $response = json_decode($this->client->textGeneration($info['siliconflow_endpoint'], $chat_input), TRUE);
    }
    catch (\Exception $e) {
      // If the rate limit is reach, special error.
      if (strpos($e->getMessage(), 'Rate limit reached') !== FALSE) {
        throw new AiRateLimitException($e->getMessage());
      }
      throw $e;
    }
    // We remove the inputted text.
    $message = new ChatMessage('', str_replace($chat_input, '', $response['choices'][0]['message']['content']));
    return new ChatOutput($message, $response, []);
  }

  /**
   * {@inheritdoc}
   */
  public function embeddings(string|EmbeddingsInput $input, string $model_id, array $tags = []): EmbeddingsOutput {
    $info = $this->getModelInfo('embeddings', $model_id);
    if (!$info['siliconflow_endpoint']) {
      throw new AiMissingFeatureException('Siliconflow endpoint is missing.');
    }
    $this->loadClient();
    // Normalize the input if needed.
    if ($input instanceof EmbeddingsInput) {
      $input = $input->getPrompt();
    }
    // Send the request.
    $response = json_decode($this->client->featureExtraction($info['siliconflow_endpoint'], $input), TRUE);

    return new EmbeddingsOutput($response, $response, []);
  }

  /**
   * {@inheritdoc}
   */
  public function imageClassification(string|array|ImageClassificationInput $input, string $model_id, array $tags = []): ImageClassificationOutput {
    $info = $this->getModelInfo('image_classification', $model_id);
    if (!$info['siliconflow_endpoint']) {
      throw new AiMissingFeatureException('Siliconflow endpoint is missing.');
    }
    $this->loadClient();
    // Normalize the input if needed.
    if ($input instanceof ImageClassificationInput) {
      $input = $input->getImageFile()->getBinary();
    }
    // Store temporary file.
    $temp_file = tempnam(sys_get_temp_dir(), 'ai_image_classification');
    file_put_contents($temp_file, $input);
    // Send the request.
    $response = json_decode($this->client->imageClassification($info['siliconflow_endpoint'], $temp_file), TRUE);
    // Remove the temporary file.
    unlink($temp_file);
    $classifications = [];
    if (is_array($response)) {
      foreach ($response as $row) {
        $classifications[] = new ImageClassificationItem($row['label'], $row['score']);
      }
    }
    else {
      throw new AiResponseErrorException('Invalid response from Siliconflow.');
    }

    return new ImageClassificationOutput($classifications, $response, []);
  }

  /**
   * {@inheritdoc}
   */
  public function textToImage(string|TextToImageInput $input, string $model_id, array $tags = []): TextToImageOutput {
    $info = $this->getModelInfo('text_to_image', $model_id);
    if (!$info['siliconflow_endpoint']) {
      throw new AiMissingFeatureException('Siliconflow endpoint is missing.');
    }
    $this->loadClient();
    // Normalize the input if needed.
    if ($input instanceof TextToImageInput) {
      $input = $input->getText();
    }

    try {
      $response = json_decode($this->client->textToImage($info['siliconflow_endpoint'],$input),TRUE);
    }
    catch (\Exception $e) {
      // Try to figure out rate limit issues.
      if (strpos($e->getMessage(), 'Request too large') !== FALSE) {
        throw new AiRateLimitException($e->getMessage());
      }
      if (strpos($e->getMessage(), 'Too Many Requests') !== FALSE) {
        throw new AiRateLimitException($e->getMessage());
      }
      // Try to figure out quota issues.
      if (strpos($e->getMessage(), 'You exceeded your current quota') !== FALSE) {
        throw new AiQuotaException($e->getMessage());
      }
      else {
        throw $e;
      }
    }
    $images = [];

    if (empty($response['data'][0])) {
      throw new AiResponseErrorException('No image data found in the response.');
    }
    // Process the image response.
    foreach ($response['data'] as $data) {
      // Check if this is a gpt-image-1 response.
      $is_gpt_image = strpos($model_id, 'gpt-image') === 0 || isset($data['revised_prompt']);

      if (isset($data['b64_json'])) {
        // Determine image type based on output_format if available.
        $mime_type = 'image/png';
        $file_ext = 'png';
        if (isset($payload['output_format'])) {
          switch ($payload['output_format']) {
            case 'jpeg':
              $mime_type = 'image/jpeg';
              $file_ext = 'jpeg';
              break;

            case 'webp':
              $mime_type = 'image/webp';
              $file_ext = 'webp';
              break;
          }
        }
        $images[] = new ImageFile(base64_decode($data['b64_json']), $mime_type, ($is_gpt_image ? 'gpt-image' : 'dalle') . '.' . $file_ext);
      }
      // Try url if b64_json is not available.
      elseif (isset($data['url']) && !empty($data['url'])) {
        try {
          $image_content = file_get_contents($data['url']);
          if ($image_content !== FALSE) {
            $images[] = new ImageFile($image_content, 'image/png', ($is_gpt_image ? 'gpt-image' : 'dalle') . '.png');
          }
          else {
            $this->logger->error('Failed to fetch image from URL: @url', ['@url' => $data['url']]);
          }
        }
        catch (\Exception $e) {
          $this->logger->error('Error fetching image URL: @error', ['@error' => $e->getMessage()]);
        }
      }
      else {
        $this->logger->error('No valid image data found in response');
      }
    }

    // If no images were successfully created, throw an error.
    if (empty($images)) {
      throw new AiResponseErrorException('Failed to process any valid images from the API response.');
    }
    return new TextToImageOutput($images, $response, []);
  }

  /**
   * {@inheritdoc}
   */
  public function maxEmbeddingsInput($model_id = ''): int {
    // @todo this is playing safe. Ideally, we should provide real number per model.
    return 1024;
  }

  /**
   * {@inheritdoc}
   */
  public function loadModelsForm(array $form, $form_state, string $operation_type, string|NULL $model_id = NULL): array {
    $form = parent::loadModelsForm($form, $form_state, $operation_type, $model_id);
    $config = $this->loadModelConfig($operation_type, $model_id);

    $form['model_data']['siliconflow_endpoint'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Endpoint'),
      '#description' => $this->t('The endpoint needed to access the dedicated Siliconflow API or the model name for shared inference. You can find possible models <a href=":link" target="_blank">here</a>. Model names autocompletes.', [
        ':link' => 'https://cloud.siliconflow.cn/me/models',
      ]),
      '#default_value' => $config['siliconflow_endpoint'] ?? '',
      '#required' => TRUE,
      '#weight' => -10,
      '#autocomplete_route_name' => 'ai_provider_siliconflow.autocomplete.models',
      '#autocomplete_route_parameters' => [
        'model_type' => $this->supportedTypes[$operation_type]['filter'],
      ],
    ];

    return $form;
  }

}
